import datetime
import json
import logging
import os
import shutil
import tarfile

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr

from sandbox.projects.yabs.base_bin_task import BaseBinTask
from sandbox.projects.yabs.global_info.resources import YABS_APP_OPTIONS

from sandbox.projects.autobudget.back_to_back.lib.compress import Compressor
from sandbox.projects.autobudget.back_to_back.lib.config import (
    MYSQL_TABLES_TO_PACK,
    TESTING_FILE_CONTENTS,
)
from sandbox.projects.autobudget.back_to_back.lib.resources import (
    YabsAutobudgetMysqlBackup,
    YabsAutobudgetSetupBackToBackTablesTool,
    YabsAutobudgetYtBackup,
    YabsAutobudgetMYSQLTableDumperTool,
    YabsOldAutobudgetLxcImage,
    YabsAutobudgetYsonTableDump,
)

MAPPING_FILE = "./yt_tables_mapping.json"


class YabsOldAutobudgetBackToBackStand(BaseBinTask):
    class Parameters(BaseBinTask.Parameters):
        container = sdk2.parameters.Container(
            "Prebuilt environment",
            default_value=None,
            resource_type=YabsOldAutobudgetLxcImage,
            required=True,
        )
        baseno = sdk2.parameters.Integer(
            "Baseno to run stand for",
            required=True,
            default=127,
        )
        mysql_archive_resource = sdk2.parameters.Resource(
            "Resource with MySQL tables backup",
            default=None,
            required=True,
            resource_type=YabsAutobudgetMysqlBackup,
        )
        yt_backup_description = sdk2.parameters.Resource(
            "Resource with YT backup description",
            default=None,
            required=True,
            resource_type=YabsAutobudgetYtBackup,
        )
        host_options_resource = sdk2.parameters.Resource(
            "Resource with host-options",
            default=None,
            required=True,
            resource_type=YABS_APP_OPTIONS,
        )
        fake_update_hour = sdk2.parameters.String(
            "Update hour to pass to autobudget, %H%M%S",
            default="140000",
            required=True,
        )
        autobudget_process_count = sdk2.parameters.Integer(
            "Autobudget process count",
            default=32,
        )
        yt_proxy_for_launch = sdk2.parameters.String(
            "YT proxy to use for launch",
            default="hahn",
            required=True,
        )
        yt_proxy_for_diff = sdk2.parameters.String(
            "YT proxy to use for diff",
            default="hahn",
            required=True,
        )
        yt_base_path_for_launch = sdk2.parameters.String(
            "Base path where to place backup copy at YT",
            default="//home/autobudget/sandbox/back_to_back/stand",
            required=True,
        )
        yt_base_path_for_diff = sdk2.parameters.String(
            "Base path where to store data at YT",
            default="//home/autobudget/sandbox/back_to_back/stand",
            required=True,
        )
        table_dumper = sdk2.parameters.Resource(
            "Resource with MySQL table dumper tool",
            required=True,
            resource_type=YabsAutobudgetMYSQLTableDumperTool,
        )
        yt_backup_handler = sdk2.parameters.Resource(
            "Resource with YT environment setup tool",
            required=True,
            resource_type=YabsAutobudgetSetupBackToBackTablesTool,
        )
        testing_file_contents = sdk2.parameters.String(
            "Contents of testing file",
            default=TESTING_FILE_CONTENTS,
            multiline=True,
            required=True,
        )
        mysql_tables_to_pack = sdk2.parameters.List(
            "Result tables of autobudget to pack into resource",
            default=MYSQL_TABLES_TO_PACK,
            required=True,
        )
        with sdk2.parameters.Output:
            autobudget_succeed = sdk2.parameters.Bool(
                "Autobudget process succeed",
                required=True,
            )
            missing_tables = sdk2.parameters.List(
                "List of missing tables after run",
                required=True,
            )
            dumped_tables = sdk2.parameters.List(
                "Dumped tables resource_ids",
                required=True,
            )

        with sdk2.parameters.Group("Infrastructure parameters"):
            resource_attrs = sdk2.parameters.Dict(
                "Filter resource by",
                default={"name": "autobudget-back-to-back-binary"},
                description="Will be passed to 'attrs' search parameter",
            )

    class Requirements(BaseBinTask.Requirements):
        privileged = True

    def on_create(self):
        if self.Parameters.host_options_resource is None:
            self.Parameters.host_options_resource = YABS_APP_OPTIONS.find(state=ctr.State.READY).order(-sdk2.Resource.id).first()
        if self.Parameters.table_dumper is None:
            self.Parameters.table_dumper = YabsAutobudgetMYSQLTableDumperTool.find(state=ctr.State.READY).order(-sdk2.Resource.id).first()
        if self.Parameters.yt_backup_handler is None:
            self.Parameters.yt_backup_handler = YabsAutobudgetSetupBackToBackTablesTool.find(state=ctr.State.READY).order(-sdk2.Resource.id).first()

    def on_execute(self):
        autobudget_process_count = self.Parameters.autobudget_process_count
        baseno = self.Parameters.baseno
        fake_update_hour = self.Parameters.fake_update_hour
        host_options_resource = self.Parameters.host_options_resource
        mysql_archive_resource = self.Parameters.mysql_archive_resource
        mysql_tables_to_pack = self.Parameters.mysql_tables_to_pack
        table_dumper_archived_path = sdk2.ResourceData(sdk2.Resource[self.Parameters.table_dumper]).path
        testing_file_contents = self.Parameters.testing_file_contents
        token = sdk2.Vault.data(self.owner, "AUTOBUDGET_BACK_TO_BACK_YT_TOKEN")
        yt_backup_description = sdk2.ResourceData(sdk2.Resource[self.Parameters.yt_backup_description])
        yt_base_path_for_diff = self.Parameters.yt_base_path_for_diff
        yt_base_path_for_launch = self.Parameters.yt_base_path_for_launch
        yt_proxy_for_diff = self.Parameters.yt_proxy_for_diff
        yt_proxy_for_launch = self.Parameters.yt_proxy_for_launch
        yt_backup_handler_archived_path = sdk2.ResourceData(sdk2.Resource[self.Parameters.yt_backup_handler]).path

        fake_update_day = mysql_archive_resource.date

        with sdk2.helpers.ProgressMeter("Prepare tools"):
            with tarfile.open(str(table_dumper_archived_path)) as tarball:
                tarball.extractall()

            with tarfile.open(str(yt_backup_handler_archived_path)) as tarball:
                tarball.extractall()

        with sdk2.helpers.ProgressMeter("Setup YT environment"):
            with self.memoize_stage.generate_prefix(commit_on_entrance=False):
                yt_tables_prefix_for_launch = "{base_path}/{task_id}".format(base_path=yt_base_path_for_launch, task_id=self.id)
                yt_tables_prefix_for_diff = "{base_path}/{task_id}".format(base_path=yt_base_path_for_diff, task_id=self.id)
                self.Context.yt_tables_prefix_for_launch = yt_tables_prefix_for_launch
                self.Context.yt_tables_prefix_for_diff = yt_tables_prefix_for_diff

            yt_tables_prefix_for_launch = self.Context.yt_tables_prefix_for_launch
            yt_tables_prefix_for_diff = self.Context.yt_tables_prefix_for_diff
            logging.info("Use base path %s at %s for launch", yt_tables_prefix_for_launch, yt_proxy_for_launch)
            logging.info("Use base path %s at %s for diff", yt_tables_prefix_for_diff, yt_proxy_for_diff)
            with sdk2.helpers.ProcessLog(self, "yt_environment_setup") as pl:
                env = os.environ.copy()
                env["YT_TOKEN"] = token
                env["YT_LOG_LEVEL"] = "debug"
                sp.check_call(
                    [
                        "./setup_back_to_back_tables",
                        "--yt-proxy", yt_proxy_for_launch,
                        "--verbose",
                        "restore",
                        yt_tables_prefix_for_launch,
                        "--backup-description-file", str(yt_backup_description.path),
                        "--mapping-file", MAPPING_FILE,
                    ],
                    stdout=pl.stdout,
                    stderr=sp.STDOUT,
                    env=env,
                )

        with sdk2.helpers.ProgressMeter("Prepare mysql tables"):
            mysql_archive_path = sdk2.ResourceData(mysql_archive_resource).path
            compressor = Compressor()
            compressor.unpack(
                package_path=str(mysql_archive_path),
                destination_path="/var/lib",
            )

        with sdk2.helpers.ProgressMeter("Prepare host-options"):
            host_options_path = sdk2.ResourceData(host_options_resource).path
            sp.check_call(["mkdir", "-p", "/var/lib/global-info"])
            with tarfile.open(str(host_options_path)) as tarball:
                tarball.extractall(path="/var/lib/global-info/global-host-options")

        with sdk2.helpers.ProgressMeter("Prepare testing file"):
            with open(MAPPING_FILE) as f:
                backup_mapping = json.load(f)
            with open("/etc/testing", "w") as testing_file:
                yt_tables_mapping = [
                    "hostoption-autobudget-yt-path-of-{table_name} {table_path}".format(table_name=table_name, table_path=table_path)
                    for table_name, table_path
                    in backup_mapping.items()
                ]

                testing_file.write(
                    testing_file_contents.format(
                        baseno=baseno,
                        fake_update_day=fake_update_day,
                        fake_update_hour=fake_update_hour,
                        yt_cluster=yt_proxy_for_launch,
                        yt_token=token,
                        process_count=autobudget_process_count,
                        yt_tables_mapping="\n".join(yt_tables_mapping),
                    )
                )

        with sdk2.helpers.ProgressMeter("Setup configuration"):
            with sdk2.helpers.ProcessLog(self, logger="prepare_environment") as pl:
                sp.check_call(["mkdir", "-p", "/home/yabs/.yt"])
                with open("/home/yabs/.yt/token", "w") as token_file:
                    token_file.write(unicode(token))

                sp.check_call(
                    ["sudo", "service", "runsvdir", "stop"],
                    stdout=pl.stdout,
                    stderr=sp.STDOUT,
                )
                sp.check_call(
                    ["sudo", "service", "mysql.yabs", "restart"],
                    stdout=pl.stdout,
                    stderr=sp.STDOUT,
                )
                sp.check_call(
                    ["chown", "-R", "yabs:yabs", "/home/yabs/stat"],
                    stdout=pl.stdout,
                    stderr=sp.STDOUT,
                )
                sp.check_call(
                    ["chown", "-R", "yabs:yabs", "/var/lib/mysql.yabs"],
                    stdout=pl.stdout,
                    stderr=sp.STDOUT,
                )

        with sdk2.helpers.ProgressMeter("Run autobudget"):
            autobudget_succeed = True
            with sdk2.helpers.ProcessLog(self, logger="autobudget") as pl:
                env = {
                    "BASENO": str(baseno),
                }
                autobudget_return_code = sp.call(
                    ["sudo", "-u", "yabs", "/home/yabs/stat/autobudget.pl"],
                    env=env,
                    stdout=pl.stdout,
                    stderr=pl.stderr,
                )
                if autobudget_return_code != 0:
                    autobudget_succeed = False

        with sdk2.helpers.ProgressMeter("Pack result tables"):
            missing_tables = set()

            self.Context.upload_stats = [
                "{now} | Started to upload".format(now=datetime.datetime.now()),
            ]
            self.Context.save()
            dumped_tables = []
            for table in mysql_tables_to_pack:
                table = table.format(
                    baseno=baseno,
                )
                if os.path.exists("/var/lib/mysql.yabs/yabsdb/{table}.MYD".format(table=table)):
                    dump_resource = YabsAutobudgetYsonTableDump(
                        self,
                        "Table {table} dump".format(table=table),
                        "{table}.tar.zstd".format(table=table),
                        yt_path="{prefix}/{table}".format(prefix=yt_tables_prefix_for_diff, table=table),
                    )
                    dump_resource_data = sdk2.ResourceData(dump_resource)
                    dump_directory = self.path(table)
                    dump_directory.mkdir()
                    with sdk2.helpers.ProcessLog(self, "upload_tables.{table}".format(table=table)) as pl:
                        sp.check_call(
                            [
                                "./table_dumper",
                                "--mysql-table-name", table,
                                "--destination-directory-path", str(dump_directory),
                            ],
                            stdout=pl.stdout,
                            stderr=sp.STDOUT,
                        )

                    compressor.pack(
                        source_path="./",
                        destination_path=str(dump_resource_data.path),
                        cwd=str(dump_directory),
                    )
                    shutil.rmtree(str(dump_directory))
                    dump_resource_data.ready()
                    dumped_tables.append(dump_resource.id)
                    self.Context.upload_stats.append(
                        "{now} | Dumped table {table}".format(now=datetime.datetime.now(), table=table),
                    )
                    self.Context.save()
                else:
                    self.Context.upload_stats.append(
                        "{now} | No {table} found".format(now=datetime.datetime.now(), table=table),
                    )
                    self.Context.save()
                    missing_tables.add(table)

        self.Parameters.autobudget_succeed = autobudget_succeed
        self.Parameters.missing_tables = list(missing_tables)
        self.Parameters.dumped_tables = dumped_tables

        if not autobudget_succeed:
            raise TaskFailure("Autobudget failed")

    @sdk2.report(title="Upload stats")
    def upload_stats(self):
        if self.Context.upload_stats is not ctm.NotExists:
            return "<br>".join(self.Context.upload_stats)
        else:
            return "<img src=\"https://jing.yandex-team.ru/files/snusmumrik/2019-01-12%2001.41.35.jpg\"/>"
