# -*- coding: utf-8 -*-

from pathlib2 import Path
from sandbox import sdk2

from sandbox.projects.yabs.base_bin_task import BaseBinTask

from sandbox.projects.autobudget.back_to_back.lib.resources import YabsAutobudgetYsonTableDump
from sandbox.projects.autobudget.back_to_back.lib.compress import Compressor


class YabsAutobudgetTableUploader(BaseBinTask):
    class Parameters(BaseBinTask.Parameters):
        dump_to_upload = sdk2.parameters.Resource(
            'Resource with dump to upload',
            resource_type=YabsAutobudgetYsonTableDump,
            required=True,
        )

        with sdk2.parameters.Group("Infrastructure parameters"):
            resource_attrs = sdk2.parameters.Dict(
                "Filter resource by",
                default={"name": "autobudget-back-to-back-binary"},
                description="Will be passed to 'attrs' search parameter",
            )

    def on_execute(self):
        dump_to_upload_resource = self.Parameters.dump_to_upload
        dump_to_upload_path = sdk2.ResourceData(dump_to_upload_resource).path

        data_directory = Path("./table_data")
        data_directory.mkdir()

        compressor = Compressor()
        compressor.unpack(
            package_path=str(dump_to_upload_path),
            destination_path=str(data_directory),
        )

        destination_yt_path = dump_to_upload_resource.yt_path
        self.upload_table(data_directory, destination_yt_path)

    def upload_table(self, data_directory, destination_yt_path):
        import yt.wrapper as yt
        from yt.wrapper import yson, YtClient, TablePath

        yt.config["write_parallel"]["enable"] = True

        schema_file = list(data_directory.glob("*.schema"))[0]

        with schema_file.open() as schema_stream:
            schema = yson.load(schema_stream)

        yt = YtClient(proxy="hahn", token=sdk2.Vault.data(self.owner, "AUTOBUDGET_BACK_TO_BACK_YT_TOKEN"))

        yt.create("table", destination_yt_path, attributes={"schema": schema, "optimize_for": "scan"}, force=True, recursive=True)

        destination_yt_path = TablePath(destination_yt_path, append=True)
        chunk_files = list(data_directory.glob("*.yson"))
        with sdk2.helpers.ProgressMeter("Uploading table chunks", minval=0, maxval=len(chunk_files)) as meter:
            for chunk_file in chunk_files:
                self.upload_chunk(chunk_file, destination_yt_path)
                meter.add(1)

    def upload_chunk(self, chunk_file, destination_path):
        from yt.wrapper import YtClient

        yt = YtClient(proxy="hahn", token=sdk2.Vault.data(self.owner, "AUTOBUDGET_BACK_TO_BACK_YT_TOKEN"))

        with chunk_file.open('rb') as chunk_stream:
            yt.write_table(destination_path, chunk_stream, format="yson", raw=True)
