# -*- coding: utf-8 -*-

import datetime
import tarfile
import os

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.resource as ctr

from sandbox.projects.autobudget.back_to_back.lib.resources import (
    YabsAutobudgetSetupBackToBackTablesTool,
    YabsAutobudgetYtBackup,
)


class YabsAutobudgetPrepareYTBackup(sdk2.Task):
    """
        Prepare Autobudget database backup
    """

    class Parameters(sdk2.Parameters):
        use_date_of_execution = sdk2.parameters.Bool(
            "Set backup date on execution",
            default=True,
        )
        with use_date_of_execution.value[False]:
            backup_date = sdk2.parameters.String(
                "Backup date (YYYYMMDD)",
                default=None,
            )

        yt_base_path = sdk2.parameters.String(
            "Destination map node template",
            default="//home/autobudget/sandbox/back_to_back/backup/{task_id}",
        )

        with sdk2.parameters.Group("Internal parameters"):
            yt_backup_handler = sdk2.parameters.Resource(
                "Resource with YT environment setup tool",
                required=True,
                resource_type=YabsAutobudgetSetupBackToBackTablesTool,
            )
            yt_proxy = sdk2.parameters.String(
                "YT proxy to prepare backup at",
                default="hahn",
            )

    def on_create(self):
        if self.Parameters.yt_backup_handler is None:
            self.Parameters.yt_backup_handler = YabsAutobudgetSetupBackToBackTablesTool.find(state=ctr.State.READY).order(-sdk2.Resource.id).first()

    def on_execute(self):
        with self.memoize_stage.make_backup_date_fixed:
            if self.Parameters.use_date_of_execution:
                backup_date = datetime.datetime.now().strftime("%Y%m%d")
            else:
                backup_date = self.Parameters.backup_date
            self.Context.fixed_backup_date = backup_date

        with self.memoize_stage.yt_backup(commit_on_entrance=False):
            self.prepare_yt_backup()

    def prepare_yt_backup(self):
        backup_date = self.Context.fixed_backup_date
        token = sdk2.Vault.data(self.owner, "AUTOBUDGET_BACK_TO_BACK_YT_TOKEN")
        yt_backup_handler_archived_path = sdk2.ResourceData(sdk2.Resource[self.Parameters.yt_backup_handler]).path
        yt_base_path = self.Parameters.yt_base_path.format(task_id=self.id)
        yt_proxy = self.Parameters.yt_proxy

        with sdk2.helpers.ProgressMeter("Prepare tools"):
            with tarfile.open(str(yt_backup_handler_archived_path)) as tarball:
                tarball.extractall()

        backup_resource = sdk2.ResourceData(
            YabsAutobudgetYtBackup(
                self,
                date=backup_date,
                description="YT tables backup description",
                path="backup_description.json",
            ),
        )
        with sdk2.helpers.ProcessLog(self, "setup_yt_back_to_back_tables") as pl:
            env = os.environ.copy()
            env["YT_TOKEN"] = token
            env["YT_LOG_LEVEL"] = "debug"
            sp.check_call(
                [
                    "./setup_back_to_back_tables",
                    "--yt-proxy", yt_proxy,
                    "--verbose",

                    "backup",
                    yt_base_path,
                    "--backup-description-file", str(backup_resource.path),
                ],
                stdout=pl.stdout,
                stderr=sp.STDOUT,
                env=env,
            )
