# -*- coding: utf-8 -*-

import sandbox.common.types.task as ctt

from sandbox import common
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.common.nanny.nanny import ReleaseToNannyTask2
from sandbox.projects.porto.common.resource_types import PORTO_LAYER_SEARCH_UBUNTU_PRECISE
from sandbox.projects.yabs.base_bin_task import BaseBinTask


class YabsAutobudgetLauncher(sdk2.resource.AbstractResource):
    releasers = ["YABS_AUTOBUDDGET"]
    releasable = True


class YabsAutobudgetCloudLayer(sdk2.resource.AbstractResource):
    releasers = ["YABS_AUTOBUDDGET"]
    releasable = True


class BuildYabsAutobudget2(ReleaseToNannyTask2, BaseBinTask):
    """
        Build yabs-autobudget2 (sdk2)
    """

    WAITED_STATUSES = set(common.utils.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK))

    class Parameters(BaseBinTask.Parameters):
        with sdk2.parameters.Group("Debuilder"):
            arcadia_path = sdk2.parameters.ArcadiaUrl(
                "ArcadiaPath. Path in arcadia.yandex.ru, included debian folder. Type `svn info` and copy-paste URL",
                required=True,
            )
            arcadia_revision = sdk2.parameters.Integer(
                "Svn revision",
                required=True,
            )
            package_version = sdk2.parameters.String(
                "Package version",
                required=True,
            )

        with sdk2.parameters.Group("Porto layer"):
            parent_layer = sdk2.parameters.Resource(
                "Parent layer resource",
                default=None,
                resource_type=PORTO_LAYER_SEARCH_UBUNTU_PRECISE,
            )
            layer_type = sdk2.parameters.String(
                "Type of porto layer to build",
                default="PORTO_LAYER_YABS_AUTOBUDGET",
            )

    class Context(BaseBinTask.Context):
        binary_packages = "yabs-autobudget2"

    def on_create(self):
        if self.Parameters.parent_layer is None:
            self.Parameters.parent_layer = PORTO_LAYER_SEARCH_UBUNTU_PRECISE.find(id=2212572127).first()

    def on_execute(self):
        with self.memoize_stage.build_package:
            build_package_task_id = self.server.task({
                "children": True,
                "description": "Build yabs-autobudget2 {version}".format(version=self.Parameters.package_version),
                "notifications": [],
                "type": "YABS_DEBUILDER",
                "owner": self.owner,
                "hidden": False,
                "priority": {
                    "class": self.Parameters.priority.cls,
                    "subclass": self.Parameters.priority.cls,
                },
                "context": {
                    "ArcadiaPath": self.Parameters.arcadia_path,
                    "ArcadiaRevision": self.Parameters.arcadia_revision,
                    "PackageVersion": self.Parameters.package_version,
                    "Repositories": "yabs-precise/testing",
                    "RunBambooTests": False,
                    "CheckAlreadyUploaded": False,
                },
            })["id"]
            self.server.batch.tasks.start.update([build_package_task_id])
            self.Context.package_builder_id = build_package_task_id

            raise sdk2.WaitTask(build_package_task_id, self.WAITED_STATUSES)

        if not self.find(id=self.Context.package_builder_id, status=ctt.Status.SUCCESS).count:
            raise TaskFailure("Unsuccessful package build")

        with self.memoize_stage.build_layer:
            layer_resource = sdk2.Resource[self.Parameters.layer_type](
                self,
                "Autobudget main layer",
                "yabs_autobudget_base.tar.gz",
                arch="linux",
            )

            build_layers_task_id = self.server.task({
                "type": "BUILD_PORTO_LAYER",
                "description": "Build porto layer for yabs-autobudget2 {version}".format(version=self.Parameters.package_version),
                "owner": self.owner,
                "notifications": [],
                "children": True,
                "hidden": False,
                "priority": {
                    "class": self.Parameters.priority.cls,
                    "subclass": self.Parameters.priority.cls,
                },
                "context": {
                    "compress": "tar.gz",
                    "keep_resolv_conf": True,
                    "layer_name": "yabs_autobudget_base",
                    "layer_type": self.Parameters.layer_type,
                    "parent_layer": self.Parameters.parent_layer.id,
                    "output_resource_id": layer_resource.id,
                    "script": "\n".join(
                        (
                            "arcadia:/arc/trunk/arcadia/yabs/stat/yabs-autobudget2/setup-scripts/resolv_conf.sh",
                            "arcadia:/arc/trunk/arcadia/yabs/stat/yabs-autobudget2/setup-scripts/install_dependencies.sh",
                            "arcadia:/arc/trunk/arcadia/yabs/stat/yabs-autobudget2/setup-scripts/install_autobudget.sh",
                            "arcadia:/arc/trunk/arcadia/yabs/stat/yabs-autobudget2/setup-scripts/cleanup.sh",
                        ),
                    ),
                    "script_env": {
                        "VERSION": self.Parameters.package_version,
                    },
                },
            })["id"]
            self.server.batch.tasks.start.update([build_layers_task_id])
            self.Context.layer_builder_id = build_layers_task_id

            raise sdk2.WaitTask(build_layers_task_id, self.WAITED_STATUSES)

        if not self.find(id=self.Context.layer_builder_id, status=ctt.Status.SUCCESS).count:
            raise TaskFailure("Unsuccessful layer build")
