from sandbox import sdk2
from sandbox.projects.resource_types import AbstractResource
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.resource as ctr


class AutoBudgetDailyLimitsCalculatorBin(AbstractResource):
    releasable = True
    any_arch = True
    auto_backup = True
    releasers = ["YABS_AUTOBUDDGET"]
    release_subscribers = ["m-ustinov"]


class AutoBudgetDailyLimitsCalculator(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 2048
        disk_space = 2048

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        default_path = '//home/yabs/dict/'
        default_tmp_path = default_path + 'AutoBudget/'
        binary_id = sdk2.parameters.LastReleasedResource(
            'Auto budget daily limits calculator  binary',
            resource_type=AutoBudgetDailyLimitsCalculatorBin,
            state=(ctr.State.READY, ),
            required=True
        )
        cluster = sdk2.parameters.String('cluster group with source tables', default='hahn')
        tm_proxy = sdk2.parameters.String('transfer manager endpoint', default='http://transfer-manager.yt.yandex.net/api/v1')
        result_cluster = sdk2.parameters.String('cluster to write dest table', default='markov')
        yt_token_secret_id = sdk2.parameters.YavSecret("YT token secret id", required=True,
                                                       description='secret should contain keys: token',
                                                       default='sec-01dg9pn791asxk9xpagtqx2q2f')
        batch_size = sdk2.parameters.Integer('batch size for write in result table in strings', default=100000)
        expired_time = sdk2.parameters.Integer('expired time for temp tables in hours (default 6)', default=6)
        expiration_time = sdk2.parameters.Integer('expiration time for temp tables in days (default 65)', default=65)
        debug = sdk2.parameters.Integer('print debug info and result on the screen', default=0)
        oid = sdk2.parameters.Integer('order id for debug. Calc and print only this order result', default=0)
        orders_table = sdk2.parameters.String('orders info table path', default=default_path + 'OrderInfo')
        ab_orders_table = sdk2.parameters.String('ab orders table path', default=default_path + 'AutoBudgetOrder')
        time_targets_table = sdk2.parameters.String('order time targets table path', default='//home/direct/db/campaigns')
        holidays_table = sdk2.parameters.String('holidays table path', default=default_path + 'Holiday')
        ab_orders_daily_limits_table = sdk2.parameters.String('orders daily limits table path', default=default_path + 'AutoBudgetOrderDailyValues')
        week_history_table = sdk2.parameters.String('orders weekly limits history table path', default=default_path + 'OrderWeeklyLimitsHistory')
        ab_orders_daily_spent = sdk2.parameters.String('orders daily stat table path', default='//home/yabs/stat/AutobudgetOrderDailySpent')
        ab_coeffs_table = sdk2.parameters.String('ab coeffs table path', default=default_path + 'AutoBudgetCoeffs')
        tmp_tables_path = sdk2.parameters.String('yt path to store tmp tables', default='//home/yabs/autobudget')
        public_tables_path = sdk2.parameters.String('yt path to store public tables', default='//home/yabs/autobudget/public')
        result_table = sdk2.parameters.String('orders daily limits result table path', default=default_path + 'AutoBudgetOrderDailyValues')
        result_weekly_table = sdk2.parameters.String('orders weekly limits result table path', default=default_path + 'AutoBudgetOrderWeeklyValues')
        result_week_history_table = sdk2.parameters.String('orders weekly limits result history table path', default=default_path + 'OrderWeeklyLimitsHistory')
        week_limit_delete_disable_percent = sdk2.parameters.Integer('orders percent to disable week limit deletion with', default=0)
        yt_write_error_codes_to_retry = sdk2.parameters.List(
            'List of yt error codes to retry',
            default=[1703],
            description='int codes https://a.yandex-team.ru/search?search=DEFINE_ENUM%5C(EErrorCode,%5Eyt%2Fyt%2Fclient.*,,arcadia,,200',
        )
        yt_max_write_attempt = sdk2.parameters.Integer('Yt max write attempt for each batch write', default=5)
        yt_max_total_write_attempt = sdk2.parameters.Integer('Yt max total write attempt', default=20)
        yt_retry_first_timeout = sdk2.parameters.Float('Yt write retry first timeout', default=1)
        yt_retry_timeout_factor = sdk2.parameters.Float('Yt retry timeout backoff factor', default=1.2)
        solomon_cluster = sdk2.parameters.String('Solomon cluster', default='calculator_test')
        disable_daily_and_weekly_limits_update = sdk2.parameters.Integer('Disable updateing dynamic replicated table with Daily and Weekly limits', default=0)
        yt_pool = sdk2.parameters.String('Yt Pool', default='')
        replication_queue_name = sdk2.parameters.String('Replication queue for TM', default='')
        do_transfer = sdk2.parameters.Integer('Do tranfer Daily and Weekly limits table to slave cluster', default=1)

    def on_execute(self):
        bin_res = sdk2.ResourceData(self.Parameters.binary_id)
        env = {
            'YT_TOKEN': self.Parameters.yt_token_secret_id.data()['token'],
            'SOLOMON_TOKEN': self.Parameters.yt_token_secret_id.data()['solomon_token'],
        }
        cmd = [
            str(bin_res.path),
            '--cluster={0}'.format(self.Parameters.cluster),
            '--result-cluster={0}'.format(self.Parameters.result_cluster),
            '--batch-size={0}'.format(self.Parameters.batch_size),
            '--debug={0}'.format(self.Parameters.debug),
            '--oid={0}'.format(self.Parameters.oid),
            '--orders-table={0}'.format(self.Parameters.orders_table),
            '--ab-orders-table={0}'.format(self.Parameters.ab_orders_table),
            '--time-targets-table={0}'.format(self.Parameters.time_targets_table),
            '--holidays-table={0}'.format(self.Parameters.holidays_table),
            # TODO: remove after https://a.yandex-team.ru/review/1484609
            '--order-daily-stat={0}'.format(self.Parameters.ab_orders_daily_spent),
            '--ab-orders-daily-limits-table={0}'.format(self.Parameters.ab_orders_daily_limits_table),
            '--ab-coeffs-table={0}'.format(self.Parameters.ab_coeffs_table),
            '--tmp-tables-path={0}'.format(self.Parameters.tmp_tables_path),
            '--result-table={0}'.format(self.Parameters.result_table),
            '--result-week-table={0}'.format(self.Parameters.result_weekly_table),
            '--week-limit-delete-disable-percent={0}'.format(self.Parameters.week_limit_delete_disable_percent),
            '--yt-write-error-codes-to-retry={0}'.format(','.join(map(str, self.Parameters.yt_write_error_codes_to_retry))),
            '--yt-max-write-attempt={0}'.format(self.Parameters.yt_max_write_attempt),
            '--yt-max-total-write-attempt={0}'.format(self.Parameters.yt_max_total_write_attempt),
            '--yt-retry-first-timeout={0}'.format(self.Parameters.yt_retry_first_timeout),
            '--yt-retry-timeout-factor={0}'.format(self.Parameters.yt_retry_timeout_factor),
            '--solomon-cluster={0}'.format(self.Parameters.solomon_cluster),
            '--tm-proxy={0}'.format(self.Parameters.tm_proxy),
            '--public-tables-path={0}'.format(self.Parameters.public_tables_path),
            '--disable-daily-and-weekly-limits-update={0}'.format(self.Parameters.disable_daily_and_weekly_limits_update),
            '--yt-pool={0}'.format(self.Parameters.yt_pool),
            '--replication-queue-name={0}'.format(self.Parameters.replication_queue_name),
            '--do-transfer={0}'.format(self.Parameters.do_transfer),
        ]
        if self.Parameters.expired_time:
            cmd += ['--expired-time={0}'.format(self.Parameters.expired_time)]
        if self.Parameters.expiration_time:
            cmd += ['--expiration-time={0}'.format(self.Parameters.expiration_time)]
        if self.Parameters.week_history_table:
            cmd += ['--week-history-table={0}'.format(self.Parameters.week_history_table)]
        if self.Parameters.result_week_history_table:
            cmd += ['--result-week-history-table={0}'.format(self.Parameters.result_week_history_table)]

        with sdk2.helpers.ProcessLog(self, logger="ab_daily_limits_calc") as pl:
            sp.check_call(cmd, stdout=pl.stdout, stderr=pl.stderr, env=env)
