from multiprocessing.pool import Pool

from sandbox import sdk2
from sandbox.sandboxsdk import environments as envs
import sandbox.common.types.client as ctc


from sandbox.projects.autobudget.autobudget_lib import get_all_bstats


def get_rm_bids(connection_settings):
    import MySQLdb
    host, user, password = connection_settings
    bstat_db = MySQLdb.connect(
                host=host,
                user=user,
                passwd=password,
                db="yabsdb",
            )
    bstat_db.autocommit(True)
    cursor = bstat_db.cursor()
    cursor.execute(
        """
            select OrderID, GroupExportID, ResultCost, ResultFlatCost
            from PhrasePrice
            where ContextType = 11 and PhraseID = 11
        """,
    )
    data = cursor.fetchall()
    return data


class AutobudgetExportRmBids(sdk2.Task):
    """
        Select relevance match banners and bids from OrderInfo and write in to YT
    """
    class Parameters(sdk2.Parameters):
        dst_table = sdk2.parameters.String(
            "Destination table",
            default="//home/bs/autobudget/data/RelevanceMatchBids",
        )
        yt_token_vault = sdk2.parameters.String(
            "YT token Vault name",
            default="YT_TOKEN",
        )
        yt_proxy = sdk2.parameters.String(
            "YT proxy to use",
            default="hahn",
        )
        user = sdk2.parameters.String(
            "MySQL user",
            default="autobudget",
        )
        password_vault = sdk2.parameters.String(
            "MySQL password Vault Name",
            default="mysql_password",
        )

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.GENERIC & ~ctc.Tag.LXC
        environments = (
            envs.PipEnvironment("MySQL-python", "1.2.5", use_wheel=True),
            envs.PipEnvironment("yandex-yt"),
            envs.PipEnvironment("yandex-yt-yson-bindings-skynet"),
        )

    def on_execute(self):
        from yt.wrapper import YtClient

        yt = YtClient(
            proxy=self.Parameters.yt_proxy,
            token=sdk2.Vault.data(self.Parameters.yt_token_vault),
        )

        user = self.Parameters.user
        password = sdk2.Vault.data(self.owner, self.Parameters.password_vault)
        pool = Pool(processes=20)
        bstat_infos = [(fqdn, user, password) for fqdn in get_all_bstats().values()]
        rm_bids = pool.map_async(
            get_rm_bids,
            bstat_infos
        )

        with yt.Transaction():
            yt.remove(self.Parameters.dst_table, force=True)
            yt.create_table(
                self.Parameters.dst_table,
                attributes={
                    'schema': [
                        {'name': 'OrderID', 'type': 'uint64'},
                        {'name': 'GroupExportID', 'type': 'uint64'},
                        {'name': 'RelevanceMatchBid', 'type': 'double'},
                        {'name': 'RelevanceMatchRsyaBid', 'type': 'double'},
                    ]
                }
            )
            for data in list(rm_bids.get()):
                yt.write_table(
                    "<append=%true>" + self.Parameters.dst_table,
                    [
                        {
                            'OrderID': int(item[0]),
                            'GroupExportID': int(item[1]),
                            'RelevanceMatchBid': float(item[2]) / 1e6,
                            'RelevanceMatchRsyaBid': float(item[3]) / 1e6 or None,
                        }
                        for item in data
                    ]
                )
