import re

from sandbox import sdk2
from sandbox.common.errors import TaskFailure

from sandbox.projects.yabs.base_bin_task import BaseBinTask

RE_HOST_OPTION = re.compile(r'''(?:host_option|HostOption)\s*\(\s*['"]([^'"]*)['"]''')
RE_EXPERIMENT = re.compile(r'''(?:is_experiment_turned_on)\s*\(\s*['"]([^'"]*)['"](?:,\s*['"]([^'"]*)['"])?''')
RE_ISSUE = re.compile(r'([A-Z]+-[0-9]+)')


def iter_host_options(content):
    for match in RE_HOST_OPTION.finditer(content):
        yield match.group(1)

    for match in RE_EXPERIMENT.finditer(content):
        if match.group(2):
            yield match.group(2)
        else:
            yield 'autobudget-' + match.group(1)


def find_host_options_in_diff(diff):
    added_options = set()
    deleted_options = set()
    for line in diff.splitlines():
        if line.startswith('-'):
            deleted_options.update(iter_host_options(line))
        elif line.startswith('+'):
            added_options.update(iter_host_options(line))

    return added_options, deleted_options


def find_issues(message):
    issues = set()
    for match in RE_ISSUE.finditer(message):
        issues.add(match.group(1))

    return issues


class AutobudgetFindPerlHostOptionsInCommitDiff(BaseBinTask):
    class Parameters(BaseBinTask.Parameters):
        svn_url = sdk2.parameters.String(
            "SVN url",
            default="arcadia:/arc/trunk/arcadia/yabs/stat/yabs-autobudget2",
            required=True,
        )
        revision = sdk2.parameters.Integer(
            "SVN revision",
            required=True,
        )

    def on_execute(self):
        from startrek_client import Startrek

        svn_url = self.Parameters.svn_url
        revision = self.Parameters.revision

        token = sdk2.Vault.data('YABS_AUTOBUDDGET', 'autobudget_startrek_token')
        self.set_info('Got token from vault')
        st = Startrek(token=token, useragent='find_host_options')

        log = sdk2.svn.Arcadia.log(url=svn_url, revision_from=revision, revision_to=revision)
        if len(log) < 1:
            raise TaskFailure('Revision not found')
        elif len(log) > 1:
            raise TaskFailure('SVN is drunk and went home')

        commit_message = log[0]['msg']
        self.set_info('COMMIT MESSAGE:\n' + commit_message)

        diff = sdk2.svn.Arcadia.diff(url=svn_url, change=revision)
        # self.set_info('DIFF:\n' + diff)

        issues = sorted(find_issues(commit_message))
        self.set_info('ISSUES: ' + str(issues))

        added_options, deleted_options = find_host_options_in_diff(diff)
        new_options = added_options - deleted_options  # Modified existing options are present in both sets
        self.set_info('ADDED: ' + str(sorted(added_options)))
        self.set_info('DELETED: ' + str(sorted(deleted_options)))
        self.set_info('NEW: ' + str(sorted(new_options)))

        if new_options:
            for issue_id in issues:
                try:
                    issue = st.issues[issue_id]
                except Exception as err:
                    self.set_info(issue_id + ': ERROR\n' + str(err))
                    continue

                old_tags = set(issue.tags)
                required_tags = {'host-option-monitoring'} | {
                    'host-option:' + option
                    for option in new_options
                }
                if not required_tags.issubset(old_tags):
                    new_tags = old_tags | required_tags
                    self.set_info(issue_id + u': WILL UPDATE TAGS\nBEFORE: ' + u' '.join(sorted(old_tags)) + u'\nAFTER: ' + u' '.join(sorted(new_tags)))
                    issue.update(tags=sorted(new_tags))
                else:
                    self.set_info(issue_id + u': NO NEED TO UPDATE TAGS\nUNCHANGED: ' + u' '.join(sorted(old_tags)))
