import json
import yaml
import os
import time
from textwrap import dedent

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects import resource_types
from sandbox.projects.ads import BsmrScriptsBinary
import sandbox.sandboxsdk.svn as sdk_svn
from sandbox.projects.common.vcs import arc

import logging


def find_resource(search_conf):
    try:
        search_conf_params = json.loads(search_conf)
    except Exception as exc:
        raise ValueError('Error while loading resource search config: {}, {}'
                         .format(search_conf, exc))

    if type(search_conf_params) != list:
        params_list = [search_conf_params]
    else:
        params_list = search_conf_params

    resources = []
    for params in params_list:
        logging.info('find_resource with params: {}'.format(params))
        res = sdk2.Resource.find(**params).order(-sdk2.Resource.id).first()
        if res:
            resources.append(res)
    resources.sort(key=lambda r: r.id, reverse=True)
    logging.info('resource found: {}'.format(resources[0]))
    return resources[0]


def get_named_resources(resource_map):
    res = dict()
    for name, some_str in resource_map.iteritems():
        try:
            resource_id = int(some_str)
        except (ValueError, TypeError):
            resource = find_resource(some_str)
        else:
            resource = sdk2.Resource.find(id=resource_id).first()
        if resource is None:
            raise ValueError('resource not found: {!r}'.format(some_str))
        res[name] = sdk2.ResourceData(resource).path
    return res


def get_resource_params(resource_name, resource_type):
    resource = sdk2.parameters.Resource(
        resource_name + '_' + resource_type,
        description=resource_name + ' ' + resource_type,
        required=False,
        resource_type=[resource_types.ARCADIA_PROJECT,
                        BsmrScriptsBinary],
    )
    resource_search_config = sdk2.parameters.String(
        resource_name + "_search_config",
        description=dedent('''\
            Json with parameters of searching {}_binary.
            Example: {{"type": "BSMR_SCRIPTS_BINARY",
                        "state": "READY",
                        "attrs": {{"released": "stable"}}}}
        ''').format(resource_name),
        multiline=True
    )
    return resource, resource_search_config


def get_resource_revision(resource):
    if hasattr(resource, 'arcadia_revision'):
        return resource.arcadia_revision
    elif hasattr(resource, 'svn_revision'):
        return resource.svn_revision

    return "undefined"


class RunSupautobudgetRobotScript(sdk2.Task):
    """subj"""

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1024
        disk_space = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        program, program_search_config = get_resource_params("program", "binary")
        config, config_search_config = get_resource_params("config", "package")
        validate_config_revision = sdk2.parameters.Bool(
            'validate_config_revision',
            description='Check that config resource revision >= binary resource revision',
            default_value=False,
        )
        config_path = sdk2.parameters.String(
            'config_path', description='path to config, Arcadia trunk relative',
        )
        yt_token = sdk2.parameters.String(
            'yt_token', description="Yt token name (in sandbox vault)",
            required=True,
        )
        yt_token_owner = sdk2.parameters.String(
            'token_owner', description="Yt token owner (in sandbox vault)",
            required=True,
        )
        solomon_token_secret = sdk2.parameters.YavSecret(
            'Yav secret with Solomon token',
            required=False,
        )
        with sdk2.parameters.RadioGroup('Solomon installation') as solomon_installation:
            solomon_installation.values['preprod'] = solomon_installation.Value('Prestable', default=True)
            solomon_installation.values['prod'] = solomon_installation.Value('Production')
        cmd_tamplate = sdk2.parameters.String(
            'cmd_template',
            description='Command line template, use {binary}, {arcadia}, {conf} and {resource_name} variables',
            required=True,
            default_value='{binary} --conf {conf}',
        )
        named_resources = sdk2.parameters.Dict(
            'named_resources',
            description=dedent('''\
                Mapping resource_name => resource_id | search_config,
                resource_name can be used in command line template.
                Search_config example: same as program_search_config
            '''),
        )
        environment = sdk2.parameters.Dict(
            'environ', description='Environment of process',
        )
        format_conf = sdk2.parameters.Bool(
            'format_conf',
            description='Format config string to replace resources (use "{{" to write a "{")',
            default_value=False,
        )
        conf_patch = sdk2.parameters.String(
            'conf_patch', description='Patch to yabs.conf',
            multiline=True,
        )

    def get_resource(self, resource_name, optional=False):
        resource = getattr(self.Parameters, resource_name)
        if resource is None:
            resource_search_config = getattr(self.Parameters, resource_name + "_search_config")
            if resource_search_config:
                resource = find_resource(resource_search_config)
                if resource is None:
                    raise ValueError('resource not found: {!r}'.format(resource_search_config))
            else:
                message = 'resource or resource_search_config not specified for {}'.format(resource_name)
                if optional:
                    logging.warning(message + ', but it is optional')
                    return None, 0, 0
                else:
                    raise ValueError(message)

        resource_revision = get_resource_revision(resource)
        logging.warning(resource_name + ' resource revision: ' + resource_revision)
        return sdk2.ResourceData(resource).path, resource.id, resource_revision

    def on_execute(self):
        binary, program_resource_id, binary_revision = self.get_resource("program")
        conf, config_resource_id, config_revision = self.get_resource("config", optional=True)

        if self.Parameters.validate_config_revision and conf is not None:
            logging.info('checking that resources revision != "undefined" (attribute with revision has found)')
            assert binary_revision != "undefined"
            assert config_revision != "undefined"
            logging.info('checking that config resource revision >= binary resource revision')
            assert binary_revision <= config_revision
        else:
            logging.warning('skipping validation that config resource revision >= binary resource revision')

        resources = get_named_resources(self.Parameters.named_resources)
        if self.Parameters.config_path:
            for i in range(10):
                try:
                    conf_dir = sdk_svn.Arcadia.checkout(
                        sdk_svn.Arcadia.trunk_url(os.path.dirname(self.Parameters.config_path)),
                        "conf_dir"
                    )
                    break
                except Exception:
                    if i == 9:
                        raise
                    time.sleep(10)
            if self.Parameters.format_conf:
                conf = os.path.join(os.path.abspath(os.getcwd()), "ads.sandbox_task.conf~")
                with open(os.path.join(conf_dir, os.path.basename(self.Parameters.config_path))) as f:
                    with open(conf, "w") as f2:
                        content = f.read().format(**resources)
                        logging.warning(content)
                        f2.write(content)
            else:
                conf = os.path.join(conf_dir, os.path.basename(self.Parameters.config_path))

        context = dict(binary=binary, conf=conf)
        context.update(resources)
        with arc.Arc().mount_path(None, None, fetch_all=False) as arc_path:
            context.update(arcadia=arc_path)
            env = (self.Parameters.environment or {}).copy()
            env['SANDBOX_PROGRAM_RESOURCE_ID'] = str(program_resource_id)
            env['SANDBOX_CONFIG_RESOURCE_ID'] = str(config_resource_id)
            env['SANDBOX_OWNER'] = self.owner
            env['SANDBOX_TASK_ID'] = str(self.id)
            for env_key in env.keys():
                if env[env_key] and env[env_key].startswith('$(vault:value:'):
                    (_, _, owner, name) = env[env_key].split(':')
                    name = name[:-1]  # cut trailing ')'
                    env[env_key] = sdk2.Vault.data(owner, name)
            conf_patches = os.path.join(os.path.abspath(os.getcwd()), 'patch_conf')
            if not os.path.isdir(conf_patches):
                os.mkdir(conf_patches)
                with open(os.path.join(conf_patches, 'scriptcontrol.conf'), 'w') as out:
                    yaml.dump({'scriptcontrol2': {'enabled': False}, 'path': {'locks_dir': os.getcwd()}}, out)

                if self.Parameters.conf_patch:
                    with open(os.path.join(conf_patches, 'task.conf'), 'w') as out:
                        conf = self.Parameters.conf_patch
                        if self.Parameters.format_conf:
                            conf = conf.format(**resources)
                            logging.warning(conf)
                        out.write(conf)

            env['YT_TOKEN'] = sdk2.Vault.data(self.Parameters.yt_token_owner, self.Parameters.yt_token)
            if self.Parameters.solomon_token_secret:
                env['SOLOMON_TOKEN'] = self.Parameters.solomon_token_secret.data()['solomon_token']
            env['SOLOMON_INSTALLATION'] = self.Parameters.solomon_installation
            env['YABS_ADDITIONAL_CONF_PATH'] = conf_patches
            env['SYNCHROPHAZOTRON_PATH'] = str(self.synchrophazotron)
            if not env.get('YQL_TOKEN'):
                env['YQL_TOKEN'] = env['YT_TOKEN']

            # Merge created environment with the OS environment
            final_env = os.environ.copy()
            final_env.update(env)

            with sdk2.helpers.ProcessLog(self, logger="user_script") as pl:
                sp.check_call(
                    self.Parameters.cmd_tamplate.format(**context),
                    shell=True,
                    stdout=pl.stdout, stderr=sp.STDOUT,
                    env=final_env,
                )
