import json
import logging

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt
import sandbox.projects.common.binary_task as binary_task
import sandbox.projects.sandbox.deploy_binary_task as dbt

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common import rest
from sandbox.common import urls
from sandbox.projects.common.vcs import arc


class FailTaskWrapper(object):
    FAILED_TASK_LIMIT = 10

    def __init__(self, task_type, parent):
        self.task_type = task_type
        self.parent = parent
        self.__tasks = []
        self.__tracebacks = {}
        self.__subtasks = []
        self.__find_all_subtasks(self.parent)

    def __find_all_subtasks(self, task):
        subtasks = list(task.find(status=(ctt.Status.FAILURE, ctt.Status.EXCEPTION)).limit(self.FAILED_TASK_LIMIT))
        for subtask in subtasks:
            self.__find_all_subtasks(subtask)
        self.__subtasks.extend(subtasks)

    @property
    def tasks(self):
        if not self.__tasks:
            self.__tasks = [task for task in self.__subtasks if task.type.name == self.task_type]
        return self.__tasks

    @property
    def errors(self):
        if not self.__tracebacks:
            for task in self.tasks:
                task_traceback = task.Context.__getstate__().get('__last_error_trace')
                if task_traceback not in self.__tracebacks.values():
                    self.__tracebacks[task.id] = task_traceback

        return self.__tracebacks

    def has_errors(self):
        return bool(self.tasks)

    def get_errors_info(self):
        message = ['{task_type} failed, summary of unique errors:\n'.format(task_type=self.task_type)]
        for id, traceback in self.errors.items():
            message.append('subtask id: {id}\n'.format(id=id))
            if traceback:
                message.append(traceback + '\n')
            else:
                message.append('Task failed with unknown error\n')

        return ''.join(message)


class AutocheckAcceptance(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Requirements(sdk2.Requirements):
        pass

    class Parameters(sdk2.Parameters):
        arcadia_url = sdk2.parameters.Url('Arcadia url')
        arcadia_patch = sdk2.parameters.String('Apply patch (text diff or rbtorrent)', multiline=True)

        check_precommit = sdk2.parameters.Bool('Do integrational check with precommit parameters', default=True)
        check_postcommit = sdk2.parameters.Bool('Do integrational check with postcommit parameters', default=True)

        check_resources = sdk2.parameters.Bool('Check last 4 resources(2 stable, testing and, if exists, prestable)')

        client_tags = sdk2.parameters.ClientTags('Client tags for children tasks', default=ctc.Tag.GENERIC)

        acceptance_run_for_precommit = sdk2.parameters.Bool('Is acceptance run for precommit check')

        with sdk2.parameters.Group('Tokens') as acceptance_tokens:
            arc_token = sdk2.parameters.YavSecret('Arc token')
            sandbox_token = sdk2.parameters.YavSecret('Sandbox token (need to upload binaries to sandbox preprod)')

        bin_params = binary_task.LastBinaryReleaseParameters()

    class Context(sdk2.Context):
        _dbt_task = None
        tasks_resource_id = None
        check_tasks = []

    @property
    def preprod_client(self):
        sandbox_token = self.Parameters.sandbox_token.data()[self.Parameters.sandbox_token.default_key]
        return rest.Client(
            base_url='https://www-sandbox1.n.yandex-team.ru/api/v1.0',
            auth=sandbox_token,
            total_wait=60
        )

    @property
    def arc_token(self):
        return self.Parameters.arc_token.data()[self.Parameters.arc_token.default_key]

    def on_execute(self):
        super(AutocheckAcceptance, self).on_execute()

        if self.Parameters.check_resources:
            with self.memoize_stage.check_tasks:
                self.check_resources()

            tasks = self.server.task.read(
                id=','.join([str(id) for id in self.Context.check_tasks]),
                limit=len(self.Context.check_tasks)
            )
            bad = [task for task in tasks['items'] if task['status'] not in ctt.Status.SUCCESS]
            if bad:
                raise errors.TaskFailure('Check resource tasks failed: {}'.format(bad))
        else:
            self.build_and_test_autocheck_tasks()
            if not self.Parameters.acceptance_run_for_precommit:
                self.push_to_preprod()

    def on_release(self, params):
        params['email_notifications']['cc'].append('neksard')
        super(AutocheckAcceptance, self).on_release(params)

        logging.info('on_release params: %s', params)

        # mark the resource as stable in sandbox.preprod
        try:
            responce = self.preprod_client.resource.read(
                limit=1,
                attrs={'autocheck_tasks': True, 'origin_production_rid': self.Context.tasks_resource_id},
                state=ctr.State.READY
            )

            logging.debug('find resource in preprod: %s', responce)

            if len(responce['items']) == 0:
                logging.warning('Cannot find resource %s at preprod sandbox', self.Context.tasks_resource_id)
                return

            preprod_res_id = responce['items'][0]['id']

            self.preprod_client.resource[preprod_res_id].attribute['released'].update(name='released', value=params['release_status'])
            if params['release_status'] == ctt.ReleaseStatus.STABLE:
                if 'ttl' in responce['items'][0]['attributes']:
                    self.preprod_client.resource[preprod_res_id].attribute['ttl'].update(name='ttl', value='60')
                else:
                    self.preprod_client.resource[preprod_res_id].attribute.create(name='ttl', value='60')
        except Exception as e:
            logging.error('Cannot change resource release params in sandbox preprod:\n%s', e)

    def push_to_preprod(self):
        tasks_resource = self.server.resource[self.Context.tasks_resource_id].read()

        required_attributes = tasks_resource['attributes']
        required_attributes['backup_task'] = True
        required_attributes['origin_production_rid'] = tasks_resource['id']

        custom_fields = [
            ('resource_type', tasks_resource['type']),
            ('remote_file_protocol', 'skynet'),
            ('created_resource_name', tasks_resource['file_name']),
            ('remote_file_name', tasks_resource['skynet_id']),
            ('resource_attrs', ','.join('{}={}'.format(k, v) for k, v in required_attributes.items()))
        ]

        task_descr = {
            'type': 'REMOTE_COPY_RESOURCE',
            'description': '{}\n\nSource: {}'.format(tasks_resource['description'], urls.get_resource_link(self.Context.tasks_resource_id)),
            'owner': tasks_resource['owner'],
            'custom_fields': [{'name': name, 'value': value} for name, value in custom_fields]
        }

        logging.debug('Create REMOTE_COPY_RESOURCE with options: %s', json.dumps(task_descr, indent=4))

        try:
            task = self.preprod_client.task(task_descr)
            responce = self.preprod_client.batch.tasks.start.update(task['id'])[0]
            logging.debug('task: %s', task)
            logging.debug('responce: %s', responce)

            if responce['status'] != ctm.BatchResultStatus.SUCCESS:
                logging.warning('Fail to push autocheck tasks resource to sandbox preprod: %s', responce)
        except Exception as e:
            logging.error('Cannot push resource to sandbox preprod:\n%s', e)

    def build_and_test_autocheck_tasks(self):
        with self.memoize_stage.build_and_test_autocheck_tasks(commit_on_entrance=False):
            integrational_check = self.Parameters.check_precommit or self.Parameters.check_postcommit

            if self.Parameters.check_precommit and self.Parameters.check_postcommit:
                integrational_payload = self.tasks_parameters()
            elif self.Parameters.check_precommit:
                integrational_payload = self.precommit_parameters()
            elif self.Parameters.check_postcommit:
                integrational_payload = self.postcommit_parameters()
            else:
                integrational_payload = None

            description = '[postcommit check] Create and test autocheck binary tasks'
            attrs = {'autocheck_tasks': True, 'released': 'testing', 'taskbox_enabled': True}

            if self.Parameters.acceptance_run_for_precommit:
                description = '[precommit check] Create and test autocheck binary task'
                attrs['released'] = 'cancelled'

            deploy_binary_task_parameters = dict(
                arcadia_url=self.Parameters.arcadia_url,
                target='sandbox/projects/autocheck/AutocheckBuildParent2/AutocheckBuildParent2',
                arcadia_patch=self.Parameters.arcadia_patch,
                attrs=attrs,
                integrational_check=integrational_check,
                integrational_check_payload=integrational_payload,
                use_yt_cache=False,
                release_ttl=14,
                resource_for_parent=True,
                use_arc=True,
                arc_oauth_token=self.Parameters.arc_token,
            )

            if self.Requirements.tasks_resource:
                logging.info('Parent resource is %s', self.Requirements.tasks_resource)
                deploy_binary_task_parameters['__requirements__'] = {'tasks_resource': self.Requirements.tasks_resource.id}
                # need to set binary_executor_release_type to custom in order to run the task with the same binary resource
                deploy_binary_task_parameters['binary_executor_release_type'] = 'custom'

            task = dbt.DeployBinaryTask(
                self,
                description=description,
                **deploy_binary_task_parameters
            )
            task.enqueue()

            self.Context._dbt_task = task.id

            raise sdk2.WaitTask([self.Context._dbt_task], (ctt.Status.Group.FINISH, ctt.Status.Group.BREAK), wait_all=False)

        if sdk2.Task[self.Context._dbt_task].status not in (ctt.Status.SUCCESS, ):
            failed_apy_tasks = FailTaskWrapper('AUTOCHECK_BUILD_YA_2', self)
            failed_abp_tasks = FailTaskWrapper('AUTOCHECK_BUILD_PARENT_2', self)
            failed_dbt_tasks = FailTaskWrapper('DEPLOY_BINARY_TASK', self)
            if failed_apy_tasks.has_errors():
                self.set_info(failed_apy_tasks.get_errors_info())
            elif failed_abp_tasks.has_errors():
                self.set_info(failed_abp_tasks.get_errors_info())
            elif failed_dbt_tasks.has_errors():
                self.set_info(failed_dbt_tasks.get_errors_info())

            raise errors.TaskFailure('Fail to create resource with autocheck binary tasks')

        tasks_resource = sdk2.Resource.find(task_id=self.id, state=ctr.State.READY, attrs={'autocheck_tasks': True}).first()

        if tasks_resource:
            self.Context.tasks_resource_id = tasks_resource.id
        else:
            raise errors.TaskFailure('Autocheck tasks resource is broken')

    def check_resources(self):
        resources = []

        testing_responce = self.server.resource.read(
            limit=1,
            state=ctr.State.READY,
            attrs={'autocheck_tasks': True, 'released': ctt.ReleaseStatus.TESTING}
        )

        if not testing_responce['items']:
            raise errors.TaskError('Fail to find last testing resource')

        testing_resource = testing_responce['items'][0]['id']
        resources.append(testing_resource)
        logging.debug('Found testing task resource: %s', testing_resource)

        prestable_responce = self.server.resource.read(
            limit=1,
            state=ctr.State.READY,
            attrs={'autocheck_tasks': True, 'released': ctt.ReleaseStatus.PRESTABLE}
        )

        if prestable_responce['items']:
            resources.append(prestable_responce['items'][0]['id'])

        stable_responce = self.server.resource.read(
            limit=2,
            state=ctr.State.READY,
            attrs={'autocheck_tasks': True, 'released': ctt.ReleaseStatus.STABLE}
        )

        if len(stable_responce['items']) != 2:
            raise errors.TaskError('Fail to find last 2 stable resources')

        stable_resources = [res['id'] for res in stable_responce['items']]
        resources.extend(stable_resources)
        logging.debug('Found stable task resources: %s', stable_resources)

        tasks = []
        with rest.Batch(self.server) as api:
            for resource in resources:
                task_descr = self.precommit_parameters()[0]
                task_descr.update(
                    parent=dict(id=self.id),
                    children=True,
                    description='Check resource {}'.format(resource),
                    owner=self.owner,
                    notifications=[],
                    requirements={'tasks_resource': resource, 'client_tags': self.Parameters.client_tags},
                )
                tasks.append(api.task.create(**task_descr))

        for task in tasks:
            result = task.result()
            self.Context.check_tasks.append(result['id'])

        responce = self.server.batch.tasks.start.update(id=self.Context.check_tasks)
        if any(st['status'] == ctm.BatchResultStatus.ERROR for st in responce):
            raise errors.TaskError('Failed to start all check resource tasks')

        raise sdk2.WaitTask(self.Context.check_tasks, (ctt.Status.Group.FINISH, ctt.Status.Group.BREAK))

    def tasks_parameters(self):
        options = []
        for arcadia_url, revision, prev_arcadia_url in self._prepare_arcadia_url():
            options.extend([
                PrecommitOptions(arcadia_url, revision).to_json(),
                PostcommitOptions(arcadia_url, revision).to_json(),
                Postcommits20Options(arcadia_url, revision, prev_arcadia_url).to_json(),
                RecheckOptions(arcadia_url, revision).to_json(),
            ])
        logging.info('Integrational payload with precommit and postcommit tasks: %s', json.dumps(options, indent=2, sort_keys=True))
        return options

    def precommit_parameters(self):
        options = [PrecommitOptions(arcadia_url, revision).to_json() for arcadia_url, revision, _ in self._prepare_arcadia_url()]
        logging.info('Integrational payload with precommit task: %s', json.dumps(options, indent=2, sort_keys=True))
        return options

    def postcommit_parameters(self):
        options = []
        for arcadia_url, revision, prev_arcadia_url in self._prepare_arcadia_url():
            options.extend([
                PostcommitOptions(arcadia_url, revision).to_json(),
                Postcommits20Options(arcadia_url, revision, prev_arcadia_url).to_json(),
            ])
        logging.info('Integrational payload with postcommit task: %s', json.dumps(options, indent=2, sort_keys=True))
        return options

    def recheck_parameters(self):
        options = [RecheckOptions(arcadia_url, revision).to_json() for arcadia_url, revision, _ in self._prepare_arcadia_url()]
        logging.info('Integrational payload with recheck task: %s', json.dumps(options, indent=2, sort_keys=True))
        return options

    def _prepare_arcadia_url(self):
        arcadia_url = sdk2.vcs.svn.Arcadia.freeze_url_revision(sdk2.vcs.svn.Arcadia.trunk_url())
        try:
            revision = int(arcadia_url.split('@')[1])
        except ValueError:
            logging.exception('Unexpected value as svn revision in %s', arcadia_url)
        arc_hash, arc_revision = get_arc_hash(revision, self.arc_token)
        arc_url = 'arcadia-arc:/#{hash}'.format(hash=arc_hash)

        prev_arc_hash, prev_revision = get_arc_hash(revision - 10, self.arc_token)
        prev_arcadia_url = arcadia_url.split('@')[0] + '@' + str(prev_revision)
        prev_arc_url = 'arcadia-arc:/#{hash}'.format(hash=prev_arc_hash)
        return [(arcadia_url, revision, prev_arcadia_url), (arc_url, arc_revision, prev_arc_url)]


class TaskOptions(object):
    def to_json(self):
        custom_fields = []
        for name, value in self.parameters.items():
            custom_fields.append({
                'name': name,
                'value': value
            })
        return {
            'type': 'AUTOCHECK_BUILD_PARENT_2',
            'custom_fields': custom_fields,
        }


def get_arc_hash(svn_revision, arc_token):
    if isinstance(svn_revision, str):
        try:
            svn_revision = int(svn_revision)
        except ValueError:
            logging.error('Unexpected value as svn revision: %s', svn_revision)
            raise

    arc_client = arc.Arc(arc_oauth_token=arc_token)
    step = 0
    limit = 10
    rev_diff = 10 * 1000
    with arc_client.init_bare() as mp:
        while True:
            try:
                return arc_client.svn_rev_to_arc_hash(mp, svn_revision), svn_revision
            except arc.ArcCommandFailed as e:
                logging.warning('Cannot find arc hash for rev: %s at step %s', svn_revision, step)
                if step >= limit:
                    raise e
                if svn_revision - rev_diff > 0:
                    svn_revision -= rev_diff
                step += 1


class PrecommitOptions(TaskOptions):
    def __init__(self, arcadia_url, revision):
        self.parameters = dict(
            # Arcadia Parameters
            checkout_arcadia_from_url=arcadia_url,
            arcadia_patch='zipatch:https://storage-int.mds.yandex.net/get-arcadia-review/1139045/a54ff5afd8a8bb26ac9dd595cf9d8c407f57e0d4/1973938.zipatch',
            use_in_compare_repo_state=True,
            checkout_arcadia_from_url_for_in_compare_repo_state=arcadia_url,
            arcadia_patch_for_in_compare_repo_state='',
            use_custom_context=True,
            autocheck_revision=revision,
            is_trunk=True,

            # Build parameters
            json_prefix='',
            autocheck_config_path='',
            use_autocheck_yaml=True,
            autocheck_config_name='linux',
            autocheck_config_revision=revision,
            autocheck_build_type='Release',
            autocheck_ymake_rebuild=False,
            autocheck_build_vars='',
            autocheck_toolchain_new='',
            host_platform='',
            host_platform_flags='ALLOCATOR=LF;USE_EAT_MY_DATA=yes',
            target_platforms='',
            toolchain_transforms={},
            ignored_toolchains=[],
            recheck=False,
            autocheck_make_only_dirs='autocheck/tests/test_autocheck',
            custom_targets_list='',
            projects_partitions_count=2,
            cache_namespace='AC',
            use_ymake_cache='parser',
            used_ymake_cache_kind='parser',  # TODO: Will be removed soon YA-27
            ymake_cache_kind='parser_json',
            use_imprint_cache=True,
            with_profile=False,
            sandboxing=False,
            build_left_right_graph=False,
            add_changed_ok_configures=True,

            # Test parameters
            run_tests=False,
            report_skipped_suites=True,
            report_skipped_suites_only=False,
            do_not_download_tests_results=True,
            backup_tests_results=True,
            cache_tests=True,
            tests_retries=1,
            test_sizes=['small', 'medium'],
            test_size_timeout={},
            test_type_filters=[],
            merge_split_tests=False,
            remove_result_node=False,

            # DistBuild parameters
            distbuild_fixed_priority=-1,
            coordinators_filter_map='',
            pool_name='autocheck/precommits/public',
            gg_pool_name='autocheck/gg/precommits/public',
            make_context_on_distbuild_requirements_map='',
            distbs_max_queue_position=400,
            use_distbuild_nodes_statistics=False,
            autocheck_distbs_testing_cluster=False,
            autocheck_distbs_testing_cluster_id='',
            distbs_pool='',

            # Streaming parameters
            use_streaming=False,
            streaming_backends='',
            streaming_check_id='',
            emulate_streaming=False,
            report_to_ci=False,
            ci_logbroker_topic='',
            ci_logbroker_source_id='',
            ci_logbroker_partition_group=0,
            ci_check_id='',
            ci_iteration_number=1,
            ci_task_id='',

            # Task parameters
            is_precommit=True,
            is_autocheck_20=False,
            use_dist_diff=True,
            autocheck_logs_resource_ttl=3,
            keep_all_logs_and_files=False,
            save_graph_and_context=False,
            use_strace=False,
            report_namespace='',
            semaphores='[]',
            acceptance_task=True,
            commit_author='robot-autocheck-ci',
            send_logs_to_logbroker=True,
            circuit_type='full',
            native_build_targets='',
            ya_bin_token='',
            arc_token='',
            save_meta_graphs=False,  # XXX: ggaas tests do not like resources from integrational tests
            binary_executor_release_type='custom',
        )


class PostcommitOptions(PrecommitOptions):
    def __init__(self, arcadia_url, revision):
        super(PostcommitOptions, self).__init__(arcadia_url, revision)
        postcommit_parameters = dict(
            arcadia_patch='',
            use_in_compare_repo_state=False,
            checkout_arcadia_from_url_for_in_compare_repo_state='',
            arcadia_patch_for_in_compare_repo_state='',
            tests_retries=2,
            pool_name='autocheck/postcommits/public',
            is_precommit=False,
        )
        self.parameters.update(postcommit_parameters)


class Postcommits20Options(PrecommitOptions):
    def __init__(self, arcadia_url, revision, prev_arcadia_url):
        super(Postcommits20Options, self).__init__(arcadia_url, revision)
        parameters = dict(
            arcadia_patch='',
            checkout_arcadia_from_url_for_in_compare_repo_state=prev_arcadia_url,
            pool_name='autocheck/postcommits/public',
            is_precommit=False,
            is_autocheck_20=True,
        )
        self.parameters.update(parameters)


class RecheckOptions(PrecommitOptions):
    def __init__(self, arcadia_url, revision):
        super(RecheckOptions, self).__init__(arcadia_url, revision)
        recheck_parameters = dict(
            autocheck_make_only_dirs='devtools/dummy_arcadia/hello_world',
            custom_targets_list='[["devtools/dummy_arcadia/hello_world", "default-linux-x86_64-relwithdebinfo", 0]]',
        )
        self.parameters.update(recheck_parameters)
