# -*- coding: utf-8 -*-
import collections
import datetime as dt
import json
import logging
import os
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt
import traceback

from sandbox import sdk2
from sandbox.common import config
from sandbox.projects.common import binary_task
from sandbox.sandboxsdk import environments

from sandbox.projects.autocheck.lib import calc_dist_priority
from sandbox.projects.autocheck.lib import pool_constructor as pc
from sandbox.projects.autocheck.lib import profile
from sandbox.projects.autocheck.lib.core import betatesters as bt
from sandbox.projects.autocheck.lib.core import experiments as exp
from sandbox.projects.autocheck.lib.core import repo_adapter
from sandbox.projects.autocheck.lib.core import resources
from sandbox.projects.autocheck.lib.core import tasks_resource_manager

import autocheck_tokens
import partition_options
import subtasks_manager


class Defaults(object):
    MAX_SUBTASK_COUNT = 60
    DISTBS_MAX_QUEUE_POSITION = 400
    DEFAULT_DIST_PRIORITY = -200 * 1000 * 1000
    MAX_RESOURCE_PER_PARTITION_COUNT = 1001


class AutocheckBuildParent2(binary_task.LastRefreshableBinary, sdk2.Task):
    # Holidays list — date periods in which the task will not require pessimization semaphore (CI-2082)
    period_t = collections.namedtuple("period_t", "since till")
    NO_PESSIMIZATION_PERIODS = (period_t(dt.date(2021, 05, 01), dt.date(2021, 05, 10)),)

    subtask_manager = None

    class Requirements(sdk2.Requirements):
        cores = 1
        disk_space = 1024
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Arcadia parameters') as arcadia_parameters:
            checkout_arcadia_from_url = sdk2.parameters.Url('Url for arcadia', default='arcadia:/arc/trunk/arcadia')
            arcadia_patch = sdk2.parameters.String('Apply patch (text diff or rbtorrent)', multiline=True)
            # these params make sense only in autocheck 2.0 mode
            use_in_compare_repo_state = sdk2.parameters.Bool('Use comparing repo state')
            with use_in_compare_repo_state.value[True]:
                checkout_arcadia_from_url_for_in_compare_repo_state = sdk2.parameters.Url('Url for arcadia to compare with', required=False)
                arcadia_patch_for_in_compare_repo_state = sdk2.parameters.String('Apply patch (text diff or rbtorrent) for arcadia to compare with', multiline=True)
                use_custom_context = sdk2.parameters.Bool('Use custom context', default=True)
            autocheck_revision = sdk2.parameters.String('autocheck_revision')
            is_trunk = sdk2.parameters.Bool('Trunk repository', default=False)

        with sdk2.parameters.Group('Build parameters') as build_parameters:
            json_prefix = sdk2.parameters.String('Prefix for graph uids')
            autocheck_config_path = sdk2.parameters.String('Path to config')
            use_autocheck_yaml = sdk2.parameters.Bool('Use autocheck.yaml to configure partitions')
            with use_autocheck_yaml.value[True]:
                autocheck_config_name = sdk2.parameters.String('Config name from autocheck.yaml')
                autocheck_config_revision = sdk2.parameters.String('Arc hash or svn revision from where to get autocheck.yaml file')  # type: str
            with sdk2.parameters.String('Build type') as autocheck_build_type:
                autocheck_build_type.values['Release'] = autocheck_build_type.Value('Release', default=True)
                autocheck_build_type.values['Debug'] = autocheck_build_type.Value('Debug')
            autocheck_ymake_rebuild = sdk2.parameters.Bool('Rebuild')
            autocheck_build_vars = sdk2.parameters.String('Build variables', multiline=True)
            autocheck_toolchain_new = sdk2.parameters.String('Toolchain for ya')
            host_platform = sdk2.parameters.String('Host platform')
            host_platform_flags = sdk2.parameters.String('Host platform flags')
            target_platforms = sdk2.parameters.String('Target platforms')
            toolchain_transforms = sdk2.parameters.Dict('Toolchain transforms')
            ignored_toolchains = sdk2.parameters.List('Ignored toolchains')
            # must not be used by postcommits unless they are performed in autocheck 2.0 mode
            recheck = sdk2.parameters.Bool('Recheck (run check only for changed nodes)')
            autocheck_make_only_dirs = sdk2.parameters.String('Projects to build', default='all')
            custom_targets_list = sdk2.parameters.String(
                'Custom target list: [["path", "toolchain", partition_index, ["test_type", ...]], ... ]', default='')
            projects_partitions_count = sdk2.parameters.Integer('Projects partition count', default=1)
            cache_namespace = sdk2.parameters.String('Cache namespace')
            # Incremental enabling, to remove in the future
            use_ymake_cache = sdk2.parameters.String('Ymake cache kind to use', default='parser_json')
            used_ymake_cache_kind = sdk2.parameters.String('Ymake cache kind to use', default='parser_json')  # TODO: Will be removed soon YA-27
            used_ymake_minimal_cache_kind = sdk2.parameters.String('Ymake cache kind to use in minimal cache graph')
            ymake_cache_kind = sdk2.parameters.String('Ymake cache kind for download', default='parser_json')
            use_imprint_cache = sdk2.parameters.Bool('use_imprint_cache', default=False)
            trust_cache_fs = sdk2.parameters.Bool('trust_cache_fs', default=True)
            with_profile = sdk2.parameters.Bool('Run with profiling')
            sandboxing = sdk2.parameters.Bool('Enable sandboxing')
            build_left_right_graph = sdk2.parameters.Bool('build_left_right_graph')
            add_changed_ok_configures = sdk2.parameters.Bool('add_changed_ok_configures')

        with sdk2.parameters.Group('Test parameters') as test_parameters:
            run_tests = sdk2.parameters.Bool('Run tests', default=True)
            report_skipped_suites = sdk2.parameters.Bool('Report skipped suites')
            report_skipped_suites_only = sdk2.parameters.Bool('Report skipped suites only', default=False)
            do_not_download_tests_results = sdk2.parameters.Bool('Do not download tests results')
            backup_tests_results = sdk2.parameters.Bool('Backup tests results on the distbuild')
            cache_tests = sdk2.parameters.Bool('Cache tests', default=True)
            tests_retries = sdk2.parameters.Integer('Tests retries', default=2)
            test_sizes = sdk2.parameters.List('Test sizes to run', default=['small', 'medium'])
            test_size_timeout = sdk2.parameters.Dict('Test timeouts by size')
            test_type_filters = sdk2.parameters.List('Test type filters')
            merge_split_tests = sdk2.parameters.Bool('Merge split tests')
            remove_result_node = sdk2.parameters.Bool('remove result node')

        with sdk2.parameters.Group('DistBuild parameters') as distbuild_parameters:
            distbuild_priority = sdk2.parameters.Integer('DistBuild priority', default=-1)
            distbuild_fixed_priority = sdk2.parameters.Integer('Distbuild fixed priority (higher is better)', default=-1)
            distbs_timeout = sdk2.parameters.Integer('DistBuild timeout (sec)', default=3600)
            pool_name = sdk2.parameters.String('DistBuild pool name (without tree name)')
            distbs_pool = sdk2.parameters.String('DistBuild pool fully qualified name')
            gg_pool_name = sdk2.parameters.String('Graph generation distbuild pool name (without tree name)', default='autocheck/gg/precommits/public')
            gg_distbs_pool = sdk2.parameters.String('Graph Generation distbuild pool fully qualified name')
            coordinators_filter_map = sdk2.parameters.String('Cooridantor filter map')
            make_context_on_distbuild_requirements_map = sdk2.parameters.String('Requirements map of node which generate graph')
            distbs_max_queue_position = sdk2.parameters.Integer('Disbuild max queue position', default=Defaults.DISTBS_MAX_QUEUE_POSITION)
            autocheck_distbs_testing_cluster = sdk2.parameters.Bool('Use distbs testing cluster')
            autocheck_distbs_testing_cluster_id = sdk2.parameters.String('Distbs testing cluster id')

        with sdk2.parameters.Group('Streaming parameters') as streaming_parameters:
            use_streaming = sdk2.parameters.Bool('Use streaming report')
            streaming_backends = sdk2.parameters.String('Report streaming backends')
            streaming_check_id = sdk2.parameters.String('Report streaming id')
            emulate_streaming = sdk2.parameters.Bool('Only emulate streaming for debug')

        with sdk2.parameters.Group('CI Parameters') as ci_parameters:
            report_to_ci = sdk2.parameters.Bool('report_to_ci')
            ci_logbroker_topic = sdk2.parameters.String('ci_logbroker_topic')
            ci_logbroker_source_id = sdk2.parameters.String('ci_logbroker_source_id')
            ci_check_id = sdk2.parameters.String('ci_check_id')
            ci_check_type = sdk2.parameters.String('ci_check_type', default=None)
            ci_iteration_number = sdk2.parameters.Integer('ci_iteration_number', default=1)
            ci_task_id = sdk2.parameters.String('ci_task_id')

        with sdk2.parameters.Group('Task parameters') as task_parameters:
            is_precommit = sdk2.parameters.Bool('The check is precommit (uses Autocheck 2.0)')
            is_autocheck_20 = sdk2.parameters.Bool('Run the check in autocheck 2.0 mode')
            use_dist_diff = sdk2.parameters.Bool('Calc graph and context diffs on distbuild')
            autocheck_logs_resource_ttl = sdk2.parameters.Integer('Logs resource TTL', default=1)
            keep_all_logs_and_files = sdk2.parameters.Bool('Keep all logs and files')
            save_graph_and_context = sdk2.parameters.Bool('Save graph and context for further debug')
            use_strace = sdk2.parameters.Bool('Use strace')
            report_namespace = sdk2.parameters.String('TestEnv report namespace')
            semaphores = sdk2.parameters.String('Semaphores list: [{"name": "name", "capacity": x, "weight": y ...}, ...]', default='[]')
            acceptance_task = sdk2.parameters.Bool('Acceptance task')
            commit_author = sdk2.parameters.String('Commit Author')
            send_logs_to_logbroker = sdk2.parameters.Bool('Enable sending logs to yt through logbroker')
            circuit_type = sdk2.parameters.String('Check type: fast, full, fast_and_full')
            native_build_targets = sdk2.parameters.String('Native builds in format [{"path": "path", "toolchain": "toolchain", targets: []}, ...]')
            ya_bin_token = sdk2.parameters.String('Token from Sandbox Vault for `ya` runs ("name[:owner]")')
            arc_token = sdk2.parameters.String('Token from Sandbox Vault for arc ("name[:owner]")')
            # GraphGeneration parameters
            save_meta_graphs = sdk2.parameters.Bool('Save meta graphs', default=True)
            stop_after_meta_graph = sdk2.parameters.Bool('Stop after generating metagraphs', default=False)
            stress_test = sdk2.parameters.Bool('Stress test run', default=False)

        with sdk2.parameters.Output(reset_on_restart=True):
            resource_id = sdk2.parameters.Integer('Task resource id')

        bin_params = binary_task.LastBinaryReleaseParameters()

    class Context(sdk2.Context):
        max_subtask_count = Defaults.MAX_SUBTASK_COUNT
        should_pessimize_by_nodes = False
        subtasks_info = {}
        time_metrics = {}
        __STAGES = {}
        _is_check_yp_timeout = False
        normalized_dist_priority = Defaults.DEFAULT_DIST_PRIORITY

    @property
    def binary_executor_query(self):
        return tasks_resource_manager.get_query(self)

    def calc_sandbox_score(self):
        return self._get_priority_calculator().high_cache_hit

    def _is_stress_test(self):
        # TODO remove robot-testenv after full CI migration
        return self.Parameters.commit_author == 'robot-testenv' or self.Parameters.stress_test

    def check_yp_hosts(self):
        # XXX: experiment DISTBUILD-1410
        return self.Parameters.is_precommit and self.Parameters.commit_author == 'robot-distbuild-test' and 'RIGHT' in self.Parameters.tags

    def on_enqueue(self):
        super(AutocheckBuildParent2, self).on_enqueue()

        if config.Registry().common.installation != ctm.Installation.PRE_PRODUCTION:
            self.Requirements.client_tags = (ctc.Tag.LINUX_PRECISE | ctc.Tag.LINUX_TRUSTY) & (~ctc.Tag.OXYGEN | ctc.Tag.ACCEPTANCE)
        if self.Parameters.is_precommit:
            self.Requirements.client_tags |= ctc.Tag.PORTOD
        self.Requirements.environments = (environments.SvnEnvironment(), environments.GDBEnvironment('8.0.1'), )

        self.Requirements.disk_space = self.Parameters.projects_partitions_count * (20 if self.Parameters.is_precommit else 10) * 1024

        is_arc_url = repo_adapter.is_arc_url(self.Parameters.checkout_arcadia_from_url)

        if is_arc_url:
            self.Context.autocheck_revision = self.Parameters.autocheck_revision or '-1'
            self.Context.is_trunk = self.Parameters.is_trunk
        else:
            parsed_url = sdk2.vcs.svn.Arcadia.parse_url(self.Parameters.checkout_arcadia_from_url)
            self.Context.autocheck_revision = self.Parameters.autocheck_revision or parsed_url.revision
            self.Context.is_trunk = parsed_url.trunk

        semaphores = []
        semaphore_release = ctt.Status.Group.SEMAPHORES_RELEASE

        # DEVTOOLS-3897
        is_distbuild_testing = self.Parameters.autocheck_distbs_testing_cluster

        # if the task started for stress test, then do not take common semaphores
        # semphore for stress test will be recieved from task parameter: semaphores
        task_on_local_sb = config.Registry().common.installation == ctm.Installation.LOCAL
        if not task_on_local_sb and not self._is_stress_test():
            if self.Parameters.acceptance_task:
                semaphores.append(ctt.Semaphores.Acquire(name='autocheck_acceptance'))
            elif is_distbuild_testing:
                semaphores.append(ctt.Semaphores.Acquire(name='autocheck_emulation'))
            elif self.Context.is_trunk:
                if self.Parameters.is_precommit:
                    if self.Parameters.circuit_type == 'fast':
                        semaphores.append(ctt.Semaphores.Acquire(name='precommits_for_fast_check'))
                    else:
                        semaphores.append(ctt.Semaphores.Acquire(name='precommits'))
                else:
                    semaphores.append(ctt.Semaphores.Acquire(name='autocheck'))
            elif is_arc_url:
                semaphore_name = 'precommits_for_arc_branch' if self.Parameters.is_precommit else 'autocheck_for_arc_branch'
                semaphores.append(ctt.Semaphores.Acquire(name=semaphore_name))
            else:
                semaphore_name = 'precommits_for_branch' if self.Parameters.is_precommit else 'autocheck_for_branch'
                semaphores.append(ctt.Semaphores.Acquire(name=semaphore_name))

            if not is_distbuild_testing and self.Parameters.is_precommit:
                if self.Context.should_pessimize_by_nodes:
                    now = dt.datetime.now().date()
                    if not filter(lambda _: _.since <= now <= _.till, self.NO_PESSIMIZATION_PERIODS):
                        semaphores.append(ctt.Semaphores.Acquire(name='pessimized_precommit'))

        try:
            semaphores_data = json.loads(self.Parameters.semaphores)
            assert isinstance(semaphores_data, list)
            for params in semaphores_data:
                assert isinstance(params, dict)

            if semaphores_data and not self.Context.should_pessimize_by_nodes:
                for semaphore_parameters in semaphores_data:
                    semaphores.append(ctt.Semaphores.Acquire(**semaphore_parameters))
                semaphore_release = (ctt.Status.Group.BREAK, ctt.Status.Group.FINISH)
        except Exception as ex:
            self.set_info('Exception while setting semaphores: {}.'.format(ex))

        if self.check_yp_hosts():
            # take only experiment semaphore, cause we run experiment only on yp hosts
            semaphores = [ctt.Semaphores.Acquire(name='distbuild_one_time_experiment')]
            if not self.Context._is_check_yp_timeout:
                self.Context._is_check_yp_timeout = True
                self.Parameters.kill_timeout = 43200

        if semaphores:
            self.Requirements.semaphores = ctt.Semaphores(
                acquires=semaphores,
                release=semaphore_release
            )
        self.Parameters.score = self.calc_sandbox_score()
        logging.info('%s %s has score %s', self.type, self.id, self.Parameters.score)

    def on_break(self, prev_status, status):
        self.Context.subtasks_info = {}
        if self.Context.projects_partitions_count is ctm.NotExists:
            partition_count = self.Parameters.projects_partitions_count
        else:
            partition_count = self.Context.projects_partitions_count
        for partition in range(partition_count):
            self.clear_resources(partition)
        super(AutocheckBuildParent2, self).on_break(prev_status, status)

    def on_execute(self):
        super(AutocheckBuildParent2, self).on_execute()

        self.profiler = profile.FlamegraphProfiler(self.log_path('flamegraph'))
        with self.profiler.start_profile_thread(title='main'):
            self.execute()

        _, self.Context.flamegraph_report_link = self.profiler.prepare_report(self)

    @sdk2.report(title='Flamegraph report')
    def flamegraph_report(self):
        try:
            return 'Flamegraph report: <a href="{url}">report</a>'.format(url=self.Context.flamegraph_report_link)
        except:
            logging.exception('Cannot get flamgraph report')
            return 'Error while getting flamegraph report:\n{}'.format(traceback.format_exc())

    def execute(self):
        self.log_resource.ttl = self.Parameters.autocheck_logs_resource_ttl
        self.Parameters.resource_id = self.Requirements.tasks_resource.id
        logging.info('Tasks info: %s', self.Context.subtasks_info)
        logging.info('Aquired sepmaphores: %s', self.Requirements.semaphores)

        if repo_adapter.is_arc_url(self.Parameters.checkout_arcadia_from_url):
            if not self.Parameters.autocheck_revision and (self.Parameters.distbuild_fixed_priority == -1 or not self.Parameters.pool_name):
                raise Exception('In order to user arc url, you must explicitly specify autocheck_revision or (pool_name and distbuild_fixed_priority)')
        else:
            if not self.Context.autocheck_revision:
                raise Exception('Cannot get autocheck revision. In order for the task to work with svn like url specify revision in arcadia url')

        self.priority_calculator = self._get_priority_calculator()
        if self.Parameters.is_precommit or self.Parameters.acceptance_task:
            if self.Parameters.circuit_type == 'fast':
                self.Context.normalized_dist_priority = self.priority_calculator.fast_checks
            else:
                self.Context.normalized_dist_priority = self.priority_calculator.high_cache_hit
        else:
            self.Context.normalized_dist_priority = self.priority_calculator.postcommits

        self.arc_binary_path = None
        if self.Parameters.acceptance_task:
            self.arc_binary_path = repo_adapter.get_arc_binary_from_resource(self)

        arc_token = autocheck_tokens.get_arc_token(self.Parameters.arc_token, self.owner)

        sdk2.paths.remove_path(self.path('bare_repo'))
        self.arc_adapter_factory = repo_adapter.get_arc_adapter_factory(str(self.path('bare_repo')), self.arc_binary_path, arc_token)
        with self.arc_adapter_factory:
            self.arc_adapter_factory.repository_config.prepare_repo(sdk2.svn.Arcadia.normalize_url(self.Parameters.checkout_arcadia_from_url))
            self.prepare_autocheck_config()
            self.Context.projects_partitions_count = self.Parameters.projects_partitions_count
            if self.Parameters.use_autocheck_yaml:
                self.Context.projects_partitions_count = self.autocheck_config.partitions_count
            subtasks_manager.run_subtasks(self)

    def prepare_autocheck_config(self):
        from yalibrary.platform_matcher import autocheck_configuration

        self.autocheck_config = None
        if self.Parameters.use_autocheck_yaml:
            if self.Parameters.autocheck_config_revision and self.Parameters.autocheck_config_name:
                try:
                    # TODO: Use str.isdigit()
                    int(self.Parameters.autocheck_config_revision)
                    url = 'arcadia:/arc/trunk/arcadia@' + self.Parameters.autocheck_config_revision
                except ValueError:
                    url = 'arcadia-arc:/#' + self.Parameters.autocheck_config_revision

                with self.arc_adapter_factory.make_arc_thin_adapter_factory() as thin_adapter_factory:
                    sdk2.paths.remove_path(self.path('autocheck_yaml_repo'))
                    thin_adapter = thin_adapter_factory.make_arc_adapter(str(self.path('autocheck_yaml_repo')), url, None)
                    if self.Parameters.acceptance_task and not self.Parameters.stop_after_meta_graph:
                        autocheck_yaml_path = os.path.join(thin_adapter.get_source_root(), 'autocheck/tests/test_autocheck/autocheck.yaml')
                    else:
                        autocheck_yaml_path = os.path.join(thin_adapter.get_source_root(), 'autocheck/autocheck.yaml')
                    logging.info('Parse autocheck.yaml from %s', autocheck_yaml_path)
                    with open(autocheck_yaml_path) as f:
                        autocheck_yaml_content = f.read()
                    config_parser = autocheck_configuration.ConfigurationParser(autocheck_yaml_content)
                    autocheck_configs = dict(config_parser.parse())
                    self.autocheck_config = autocheck_configs.get(self.Parameters.autocheck_config_name)
                    logging.info('Get autocheck from %s, config name: %s', autocheck_yaml_path, self.autocheck_config.name)
            else:
                if self.Parameters.autocheck_config_path:
                    self.autocheck_config = autocheck_configuration.configuration_by_config_path().get(self.Parameters.autocheck_config_path)
                    logging.info('Get autocheck from resource, config name: %s', self.autocheck_config.name)
        else:
            if self.Parameters.autocheck_config_path:
                self.autocheck_config = autocheck_configuration.configuration_by_config_path().get(self.Parameters.autocheck_config_path)
                logging.info('Get autocheck config name: %s', self.autocheck_config.name)

        if not self.autocheck_config:
            raise Exception('Cannot get configuration for autocheck.yaml')

    def _get_priority_calculator(self):
        return calc_dist_priority.get_autocheck_distbuild_priority(
            self.Parameters.distbuild_priority,
            self.Parameters.distbuild_fixed_priority,
            int(self.Context.autocheck_revision),
            self.id,
            repo_adapter.is_arc_url(self.Parameters.checkout_arcadia_from_url)
        )

    def _get_make_context_on_distbuild_requirements(self, partition):
        if self.Parameters.use_autocheck_yaml:
            return self.autocheck_config[partition].graph_generation_requirements

        try:
            make_context_on_distbuild_requirements_map = self.Parameters.make_context_on_distbuild_requirements_map
            logging.info('make_context_on_distbuild_requirements_map %s', make_context_on_distbuild_requirements_map)
            make_context_on_distbuild_requirements_dict = json.loads(make_context_on_distbuild_requirements_map) if make_context_on_distbuild_requirements_map else {}
            default_make_context_on_distbuild_requirements = make_context_on_distbuild_requirements_dict.get('default')
            custom_make_context_on_distbuild_requirements = make_context_on_distbuild_requirements_dict.get('partitions', {}).get(str(partition))
            context_requirements = custom_make_context_on_distbuild_requirements or default_make_context_on_distbuild_requirements
            return json.dumps(context_requirements) if context_requirements else {}
        except Exception:
            logging.exception(traceback.format_exc())
            return {}

    def _get_coordinators_filter(self, partition):
        if self.Parameters.use_autocheck_yaml:
            return self.autocheck_config[partition].coordinator_filter

        try:
            logging.info('coordinators_filter_map %s', self.Parameters.coordinators_filter_map)
            coordinators_filter_dict = {}
            if self.Parameters.coordinators_filter_map:
                coordinators_filter_dict = json.loads(self.Parameters.coordinators_filter_map)
            default_coordinators_filter = coordinators_filter_dict.get('default', '' if self.Parameters.autocheck_distbs_testing_cluster else 'distbuild-man')
            return coordinators_filter_dict.get('partitions', {}).get(str(partition)) or default_coordinators_filter
        except:
            logging.exception(traceback.format_exc())
            return ''

    def _get_gg_coordinators_filter(self, partition, default_coordinators_filter=''):
        if self.Parameters.is_trunk and not self.Parameters.autocheck_distbs_testing_cluster:
            if self.Parameters.use_autocheck_yaml and self.autocheck_config[partition].graph_generation_coordinators_filter:
                return self.autocheck_config[partition].graph_generation_coordinators_filter
            elif self.Parameters.is_precommit:
                return pc.Defaults.GG_PRECOMMIT_COORDINATORS_FILTER
            else:
                return pc.Defaults.GG_POSTCOMMIT_COORDINATORS_FILTER
        return default_coordinators_filter

    def _get_distbs_pool(self, partition, coordinators_filter):
        try:
            if self.Parameters.distbs_pool:
                return self.Parameters.distbs_pool

            if self.Parameters.use_autocheck_yaml and self.autocheck_config[partition].pool_tree:
                pool_tree = self.autocheck_config[partition].pool_tree
            else:
                pool_tree = None
            distbs_pool = pc.construct_pool_name(coordinators_filter, pool_tree,
                                                 self.Parameters.pool_name, self.Parameters.is_precommit, self.Parameters.autocheck_distbs_testing_cluster)
            if distbs_pool:
                if self.Context.is_trunk:
                    return distbs_pool
                else:
                    logging.warning('Skip set distbuild pool %s for branch check', distbs_pool)
            else:
                logging.error('Cant select disbuild pool name')
            return ''
        except:
            logging.exception('Cannot get distbuild pool')
            return ''

    def _get_gg_distbs_pool(self, partition, coordinators_filter):
        try:
            if self.Parameters.gg_distbs_pool:
                return self.Parameters.gg_distbs_pool

            if self.Parameters.use_autocheck_yaml and self.autocheck_config[partition].graph_generation_pool_tree:
                pool_tree = self.autocheck_config[partition].graph_generation_pool_tree
            else:
                pool_tree = None
            gg_distbs_pool = pc.construct_gg_pool_name(coordinators_filter, pool_tree, self.Parameters.gg_pool_name, self.Parameters.is_precommit)
            if gg_distbs_pool:
                if self.Context.is_trunk:
                    return gg_distbs_pool
                else:
                    logging.warning('Skip set graph generation distbuild pool %s for branch check', gg_distbs_pool)
            else:
                logging.error('Cant select graph generation disbuild pool name')
            return ''
        except:
            logging.exception('Cannot get distbuild pool for graph generation')
            return ''

    def create_task_parameters(self, partition):
        # type: (int) -> PartitionOptions
        if self.Parameters.use_autocheck_yaml:
            self.Context.autocheck_config_path = self.autocheck_config.config_path
        else:
            self.Context.autocheck_config_path = self.Parameters.autocheck_config_path

        if not self._is_stress_test():
            exp.apply_experiments(self.Parameters, self.Context, bt.is_betttester(self, self.Parameters.commit_author))
        else:
            logging.info('Task in stress test mode. Skipping experiments')

        self.Context.coordinators_filter = self._get_coordinators_filter(partition)
        logging.info('Set coordinator_filter ctx parameter for partiton %s to %s', partition, self.Context.coordinators_filter)
        self.Context.gg_coordinators_filter = self._get_gg_coordinators_filter(partition, self.Context.coordinators_filter)
        logging.info('Set gg_coordinators_filter ctx parameter for partition %s to %s', partition, self.Context.gg_coordinators_filter)
        self.Context.distbs_pool = self._get_distbs_pool(partition, self.Context.coordinators_filter)
        logging.info('Set distbs_pool ctx parameter for partiton %s to %s', partition, self.Context.distbs_pool)
        self.Context.gg_distbs_pool = self._get_gg_distbs_pool(partition, self.Context.gg_coordinators_filter)
        logging.info('Set gg_distbs_pool ctx parameter for partiton %s to %s', partition, self.Context.gg_distbs_pool)

        self.Context.check_yp_hosts = self.check_yp_hosts()
        if self.Context.check_yp_hosts:
            self.Context.coordinators_filter = 'distbuild-vla'
            self.Context.distbs_pool = '//vla/' + self.Parameters.pool_name
            logging.info('Due to experiment changed distbuild pool to %s, and coodirnators_filter to %s', self.Context.distbs_pool, self.Context.coordinators_filter)
        self.Context.make_context_on_distbuild_requirements = self._get_make_context_on_distbuild_requirements(partition)
        logging.info('Set make_context_on_distbuild_requirements ctx parameter for partiton %s to %s', partition, self.Context.make_context_on_distbuild_requirements)

        self.Context.projects_partition_index = partition
        self.Context.arc_binary_path = self.arc_binary_path
        self.Context.is_trunk = self.Context.is_trunk
        self.Context.autocheck_revision = int(self.Context.autocheck_revision)

        return partition_options.PartitionOptions.create_from_autocheck_task(self.Parameters, self.Context, False)

    def clear_resources(self, partition):
        autocheck_resources = list(resources.AutocheckLogs.find(
            task=self,
            attrs={
                'partition': partition,
            },
            state=[ctr.State.NOT_READY, ctr.State.READY, ctr.State.BROKEN],
        ).order(sdk2.Resource.id).limit(Defaults.MAX_RESOURCE_PER_PARTITION_COUNT))
        if self.Parameters.is_precommit:
            autocheck_resources = autocheck_resources[1: -1]
        else:
            autocheck_resources = autocheck_resources[: -1]
        logging.info('Remove %s AutocheckLogs resources of %s partition', len(autocheck_resources), partition)
        if autocheck_resources:
            res_ids = [res.id for res in autocheck_resources]
            logging.debug('Remove resources: %s', res_ids)
            self.server.batch.resources['delete'].update(
                id=res_ids,
                commet='Remove autocheck logs'
            )
