import copy
import json
import logging
import os
import typing as tp  # noqa: F401

from sandbox.projects.autocheck.lib import pool_constructor as pc


# set None to disable steering graph builds to one cluster
GRAPH_BUILD_COORDINATORS_FILTER_DEFAULT = 'distbuild-sas-00'
GRAPH_BUILD_COORDINATORS_FILTER_FOR_PRECOMMITS = 'distbuild-sas-01'  # XXX: remove this after 18.07.2022
GRAPH_BUILD_POOL_NAME_DEFAULT = pc.Defaults.DEFAULT_PRECOMMIT_POOL_NAME  # TODO: Temporary


class PartitionOptionsError(Exception):
    pass


class PartitionOptions(object):
    def __init__(self, parameters, is_comparing_repo):
        # type: (AutocheckBuildParent2.Parameters, bool)
        self.acceptance_task = parameters.acceptance_task
        self.add_changed_ok_configures = parameters.add_changed_ok_configures
        self.arc_binary_path = parameters.arc_binary_path
        self.arc_token = parameters.arc_token
        self.arcadia_patch = parameters.arcadia_patch
        self.arcadia_patch_for_in_compare_repo_state = parameters.arcadia_patch_for_in_compare_repo_state
        self.autocheck_logs_resource_ttl = parameters.autocheck_logs_resource_ttl
        self.autocheck_revision = parameters.autocheck_revision
        self.autocheck_toolchain_new = parameters.autocheck_toolchain_new
        self.backup_tests_results = parameters.backup_tests_results
        self.build_left_right_graph = parameters.build_left_right_graph
        self.build_type = parameters.autocheck_build_type
        self.build_vars = parameters.autocheck_build_vars
        self.cache_namespace = parameters.cache_namespace
        self.cache_tests = parameters.cache_tests
        self.check_yp_hosts = parameters.check_yp_hosts
        self.checkout_arcadia_from_url = parameters.checkout_arcadia_from_url
        self.checkout_arcadia_from_url_for_in_compare_repo_state = parameters.checkout_arcadia_from_url_for_in_compare_repo_state
        self.ci_check_id = parameters.ci_check_id
        self.ci_check_type = parameters.ci_check_type
        self.ci_iteration_number = parameters.ci_iteration_number
        self.ci_logbroker_source_id = parameters.ci_logbroker_source_id
        self.ci_logbroker_topic = parameters.ci_logbroker_topic
        self.ci_task_id = parameters.ci_task_id
        self.circuit_type = parameters.circuit_type
        self.commit_author = parameters.commit_author
        self.config_path = parameters.autocheck_config_path
        self.coordinators_filter = parameters.coordinators_filter
        self.custom_targets_list = parameters.custom_targets_list
        self.distbs_max_queue_position = parameters.distbs_max_queue_position
        self.distbs_pool = parameters.distbs_pool
        self.distbs_timeout = parameters.distbs_timeout
        self.distbuild_fixed_priority = parameters.distbuild_fixed_priority
        self.distbuild_priority = parameters.distbuild_priority
        self.distbuild_testing_cluster_id = parameters.autocheck_distbs_testing_cluster_id
        self.do_not_download_tests_results = parameters.do_not_download_tests_results
        self.download_artifacts = True
        self.emulate_streaming = parameters.emulate_streaming
        self.gg_coordinators_filter = parameters.gg_coordinators_filter
        self.gg_distbs_pool = parameters.gg_distbs_pool
        self.host_platform = parameters.host_platform
        self.host_platform_flags = parameters.host_platform_flags
        self.ignored_toolchains = parameters.ignored_toolchains
        self.is_autocheck_20 = parameters.is_autocheck_20
        self.is_comparing_repo = is_comparing_repo
        self.is_precommit = parameters.is_precommit
        self.is_trunk = parameters.is_trunk
        self.json_prefix = parameters.json_prefix
        self.keep_all_logs_and_files = parameters.keep_all_logs_and_files
        self.make_context_on_distbuild = True
        self.make_context_on_distbuild_requirements = parameters.make_context_on_distbuild_requirements
        self.merge_split_tests = parameters.merge_split_tests
        self.native_build_targets = parameters.native_build_targets
        self.no_tests_on_distbuild = False
        self.normalized_dist_priority = parameters.normalized_dist_priority
        self.partition_index = parameters.projects_partition_index
        self.partitions_count = parameters.projects_partitions_count
        self.rebuild = parameters.autocheck_ymake_rebuild
        self.recheck = parameters.recheck
        self.remove_result_node = parameters.remove_result_node
        self.report_namespace = parameters.report_namespace
        self.report_skipped_suites = parameters.report_skipped_suites
        self.report_skipped_suites_only = parameters.report_skipped_suites_only
        self.report_to_ci = parameters.report_to_ci
        self.run_tests = parameters.run_tests
        self.sandboxing = parameters.sandboxing
        self.save_graph_and_context = parameters.save_graph_and_context
        self.save_meta_graphs = parameters.save_meta_graphs
        self.send_logs_to_logbroker = parameters.send_logs_to_logbroker
        self.should_pessimize_by_nodes = parameters.should_pessimize_by_nodes
        self.stop_after_meta_graph = parameters.stop_after_meta_graph
        self.stream_partition = None
        self.streaming_backends = ['http://ci-sp.yandex.net/api/v1.0']
        self.streaming_id = parameters.streaming_check_id
        self.target_platforms = parameters.target_platforms
        self.targets = parameters.autocheck_make_only_dirs
        self.test_size_timeout = parameters.test_size_timeout
        self.test_sizes = parameters.test_sizes
        self.test_type_filters = parameters.test_type_filters
        self.tests_retries = parameters.tests_retries
        self.toolchain_transforms = parameters.toolchain_transforms
        self.trust_cache_fs = parameters.trust_cache_fs
        self.use_custom_context = parameters.use_custom_context
        self.use_dist_diff = parameters.use_dist_diff
        self.use_distbuild_testing_cluster = parameters.autocheck_distbs_testing_cluster
        self.use_imprint_cache = parameters.use_imprint_cache
        self.use_in_compare_repo_state = parameters.use_in_compare_repo_state
        self.use_strace = parameters.use_strace
        self.use_streaming = parameters.use_streaming
        self.use_ymake_cache = parameters.use_ymake_cache
        # self.used_ymake_cache_kind = parameters.used_ymake_cache_kind
        self.used_ymake_minimal_cache_kind = parameters.used_ymake_minimal_cache_kind
        self.with_profile = parameters.with_profile
        self.ya_bin_token = parameters.ya_bin_token
        self.ymake_cache_kind = parameters.ymake_cache_kind

        self.postprocess()

    @property
    def arcadia_url(self):
        if self.is_comparing_repo:
            return self.checkout_arcadia_from_url_for_in_compare_repo_state
        else:
            return self.checkout_arcadia_from_url

    @property
    def arc_patch(self):
        if self.is_comparing_repo:
            return self.arcadia_patch_for_in_compare_repo_state
        else:
            return self.arcadia_patch

    @staticmethod
    def create_from_autocheck_task(parameters, context, is_comparing_repo):
        # type: (sdk2.Parameters, sdk2.Context, bool) -> PartitionOptions
        '''
        Mine partition parameters from tasks Parameters and Context.
        Parameters from tasks Parameters will be rewritten by parameters from tasks Context.
        '''
        class Params(object):
            pass

        params = Params()
        for k, v in iter(parameters):
            setattr(params, k, v)

        for k, v in iter(context):
            setattr(params, k, v)

        opts = PartitionOptions(params, is_comparing_repo)
        return opts

    def get_comparing_repo_options(self):
        comparing_repo_options = copy.deepcopy(self)
        comparing_repo_options.is_comparing_repo = True
        return comparing_repo_options

    def as_dict(self):
        return dict((k, v) for k, v in self.__dict__.items() if not k.startswith('_'))

    def postprocess(self):
        if not isinstance(self.build_vars, list):
            self.build_vars = filter(None, self.build_vars.split(';'))
        if not isinstance(self.host_platform_flags, list):
            self.host_platform_flags = filter(None, self.host_platform_flags.split(';'))
        if not isinstance(self.targets, list):
            self.targets = filter(None, self.targets.split(';'))
        if not isinstance(self.target_platforms, list):
            self.target_platforms = filter(None, self.target_platforms.split(';'))

        if self.partitions_count > 1:
            self.stream_partition = self.partition_index
            self.build_vars += [
                'RECURSE_PARTITIONS_COUNT={}'.format(self.partitions_count),
                'RECURSE_PARTITION_INDEX={}'.format(self.partition_index),
            ]

        if not isinstance(self.custom_targets_list, list):
            custom_targets_list = json.loads(self.custom_targets_list) if self.custom_targets_list else []
            self.set_custom_targets_list(custom_targets_list)

        if self.cache_namespace != "AC":
            self.save_meta_graphs = False
            logging.info("Saving meta graphs disabled because cache_namespace:%s", self.cache_namespace)
        if self.recheck:
            self.save_meta_graphs = False
            logging.info("Saving meta graphs disabled because recheck:%s", self.recheck)

    def set_custom_targets_list(self, custom_targets_list, need_for_recheck=True):
        self.custom_targets_list = []
        self.custom_recheck = False

        if custom_targets_list:
            self.custom_recheck = need_for_recheck
            for item in custom_targets_list:
                target, toolchain = item[0], item[1]
                if len(item) == 2:  # no partition
                    partition = 0
                else:
                    partition = item[2]
                test_types = [None]
                if len(item) == 4:
                    test_types = item[3]

                if partition == self.partition_index:
                    self.custom_targets_list.append([target, toolchain, test_types])

    def load_config(self, source_root):
        if not self.config_path:
            logging.info('Path to config is not defined')
            return

        config_path = os.path.join(source_root, self.config_path)
        if not os.path.exists(config_path) or not os.path.isfile(config_path):
            logging.info('Path %s for config is incorrect', config_path)
            raise PartitionOptionsError('Unable to read config')

        try:
            with open(config_path) as fp:
                config = json.load(fp)

            logging.debug('Config is loaded, %s', json.dumps(config, indent=4, sort_keys=True))

            # XXX: rename partition options to tasks parameters
            config_partition_names_conversion = {
                'autocheck_build_type': 'build_type',
                'autocheck_build_vars': 'build_vars',
                'autocheck_config_path': 'config_path',
                'autocheck_distbs_testing_cluster': 'use_distbuild_testing_cluster',
                'autocheck_distbs_testing_cluster_id': 'distbuild_testing_cluster_id',
                'autocheck_make_only_dirs': 'targets',
                'autocheck_ymake_rebuild': 'rebuild',
                'projects_partition_index': 'partition_index',
                'projects_partitions_count': 'partitions_count',
                'streaming_check_id': 'streaming_id',
            }

            for k, v in config.items():
                setattr(self, config_partition_names_conversion.get(k, k), v)

            self.postprocess()
        except Exception as e:
            logging.exception('Unable to read config %s', config_path)
            raise PartitionOptionsError('Unable to read config, {}'.format(repr(e)))

    def make_ya_options(self, ya_cmd, source_root, build_root, output_dir, artifacts_output_dir, **kwargs):
        from yalibrary.ya_helper.ya_utils import ya_options
        # TODO: Extract defaults from kwargs.get(..., DEFAULT) and make YaMakeOptions from them
        # TODO: Make some YaMakeOptions from self.something s attributes
        # TODO: Just use YaMakeOptions(..., **kwargs)
        # TODO: Summarize all of this
        options = ya_options.YaMakeOptions(
            ya_bin=ya_cmd,

            be_verbose=kwargs.get('be_verbose', True),
            no_report=True,
            with_profile=self.with_profile,
            error_file=kwargs.get('error_file'),

            save_statistics=kwargs.get('save_statistics', True),
            dump_profile=os.path.join(output_dir, 'profile.json'),
            dump_stages=os.path.join(output_dir, 'stages.json'),
            source_root=source_root,
            build_type=self.build_type.lower(),
            keep_build=kwargs.get('keep_build', True),
            build_root=build_root,
            output_dir=artifacts_output_dir,
            targets=self.targets,
            rebuild=self.rebuild,

            build_vars=self.build_vars,
            host_platform=self.host_platform,
            host_platform_flags=self.host_platform_flags,
            target_platforms=self.target_platforms,
            target_platform_ignore_recurses=self.custom_recheck,
            run_tests=self.run_tests,
            test_sizes=self.test_sizes,
            test_size_timeout=self.test_size_timeout,
            test_type_filters=self.test_type_filters,
            report_skipped_suites=self.report_skipped_suites,
            strip_idle_build_results=False,

            use_distbuild=True,

            use_distbuild_testing_cluster=self.use_distbuild_testing_cluster,
            distbuild_testing_cluster_id=self.distbuild_testing_cluster_id,
            distbs_timeout=self.distbs_timeout,
            distbs_pool=kwargs.get('distbs_pool'),
            build_custom_json=None,
            sandboxing=self.sandboxing,
            custom_context=kwargs.get('custom_context', None),
            dump_json_graph=kwargs.get('dump_json_graph', False),
            dump_distbuild_rpc_log=os.path.join(output_dir, 'distbuild_rpc_log.log'),
            dump_distbuild_result=os.path.join(output_dir, 'distbs_result.json'),
            dump_distbuild_graph=os.path.join(output_dir, 'distbs_graph.zlib'),
            dist_priority=self.normalized_dist_priority,
            # graph_gen_dist_priority=None,  set from env YA_GRAPH_GEN_DIST_PRIORITY

            build_threads=kwargs.get('build_threads'),

            misc_build_info_dir=output_dir,

            revision=kwargs.get('revision'),
            svn_url=kwargs.get('svn_url'),
            arc_url_as_working_copy_in_distbuild=kwargs.get('arc_url_as_working_copy_in_distbuild'),

            make_context_on_distbuild=self.make_context_on_distbuild,
            make_context_only=False,
            make_context_on_distbuild_requirements=self.make_context_on_distbuild_requirements,
            use_svn_repository_for_dist_build=kwargs.get('use_svn_repository_for_dist_build', False),
            download_artifacts=self.download_artifacts,
            output_only_tests=kwargs.get('output_only_tests', True),

            do_not_download_tests_results=self.do_not_download_tests_results,
            backup_tests_results=self.backup_tests_results,

            json_prefix=self.json_prefix,
            coordinators_filter=kwargs.get('coordinators_filter', None),
            cache_namespace=self.cache_namespace,
            no_tests_on_distbuild=self.no_tests_on_distbuild,
            patch_spec=kwargs.get('patch_spec'),
            report_config=self.config_path,

            testenv_report_dir=output_dir,
            stream_partition=self.stream_partition,
            streaming_task_id=kwargs.get('task_id'),  # XXX: part of ci proto message, be sure to send it

            report_skipped_suites_only=self.report_skipped_suites_only,
            build_graph_result_dir=None,
            build_graph_source_root_pattern=None,

            custom_targets_list=self.custom_targets_list,
            cache_tests=self.cache_tests,
            tests_retries=self.tests_retries,
            toolchain_transforms=self.toolchain_transforms,

            # dont_merge_split_tests=None,  set from env YA_MERGE_SPLIT_TESTS
            remove_result_node=self.remove_result_node,
            # strip_packages_from_results=None, set from env YA_STRIP_PACKAGES_FROM_RESULTS

            build_threads_for_distbuild=kwargs.get('build_threads_for_distbuild'),

            # profile_to=None, NEW

            skip_test_console_report=True,
            ignore_nodes_exit_code=True,
            warning_mode=['dirloops', 'ChkPeers', 'allbadrecurses'],

            add_modules_to_results=True,
            do_not_output_stderrs=True,
            add_changed_ok_configures=self.add_changed_ok_configures,
        )

        if kwargs.get('enable_streaming_diractly_to_ci'):
            if self.report_to_ci:
                check_type = self.ci_check_type
                if not check_type:
                    if self.circuit_type == 'fast':
                        check_type = 'FAST'
                    else:
                        check_type = 'FULL'
                options += ya_options.YaMakeOptions(
                    report_to_ci=self.report_to_ci,
                    ci_topic=self.ci_logbroker_topic,
                    ci_source_id='{source_id}/{partition}'.format(source_id=self.ci_logbroker_source_id, partition=self.partition_index),
                    ci_check_id=self.ci_check_id,
                    ci_check_type=check_type,
                    ci_iteration_number=self.ci_iteration_number,
                    ci_task_id_string=self.ci_task_id,
                    ci_logbroker_partition_group=kwargs.get('ci_logbroker_partition_group'),
                    ci_logbroker_token=kwargs.get('ci_logbroker_token'),
                )
            else:
                options += ya_options.YaMakeOptions(
                    streaming_url=self.streaming_backends[0],
                    streaming_id=self.streaming_id,
                )
        else:
            options += ya_options.YaMakeOptions(
                streaming_url=kwargs.get('streaming_url'),
                streaming_id=self.streaming_id,
            )

        return options
