# -*- coding: utf-8 -*-

import calendar
import copy
import enum
import json
import logging
import multiprocessing
import os
import sandbox.common.types.task as ctt
import six
import time
import traceback
import typing as tp  # noqa
from sandbox import sdk2
from sandbox.common import errors
from sandbox.common import log
from sandbox.projects.common import error_handlers as eh
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import util as sdk_util
from sandbox.sdk2 import paths as sdk_paths
from sandbox.sdk2 import svn

import autocheck_tokens
from sandbox.projects.autocheck.lib import native_builds
from sandbox.projects.autocheck.lib.builders import ymake
from sandbox.projects.autocheck.lib.core import errors as autocheck_errors
from sandbox.projects.autocheck.lib.core import resource_packager
from sandbox.projects.autocheck.lib.core import resources
from sandbox.projects.autocheck.lib.reports import stream, yt_report
from sandbox.projects.autocheck.lib.reports import unified_agent_wrapper

# TODO(yazevnul or snowball): better
MEDIUM_CACHE_HIT_GRAPH_THRESHOLD = 10 * 1000
LOW_CACHE_HIT_GRAPH_THRESHOLD = 40 * 1000


def enable_streaming_diractly_to_ci(opts):
    # return opts.is_precommit and opts.commit_author == 'neksard'
    return False


class BuildMode(enum.Enum):
    ONLY_BUILD = 1
    GPAPH_AND_BUILD = 2


class BuildState(object):
    def __init__(self, options):
        self.start_time = None
        self.receivers = []
        self.report_server = None
        self.report_server_url = None

        self.meta_graph_result = None
        self.context_path = None
        self.uids_for_affected_nodes = None
        self.uids = None

        self.build_graph_stages_path = None
        self.build_graph_profile_path = None
        self.build_graph_last_stage = None

        self.native_builds = None

    @property
    def have_fallback_meta_graph(self):
        return self.meta_graph_result and os.path.exists(self.meta_graph_result.fallback.meta_graph_path)

    @property
    def have_minimal_cache_meta_graph(self):
        minimal_cache_graph = self.meta_graph_result and self.meta_graph_result.minimal_cache
        return minimal_cache_graph and os.path.exists(minimal_cache_graph.meta_graph_path)


class Paths(object):
    def __init__(self, room_path):
        sdk_paths.make_folder(room_path, delete_content=True)
        self.room_path = room_path
        self.source_root = None

    def setup(self, real_source_root):
        logging.debug(
            'Setup paths room_path=%s source_root=%s',
            self.room_path,
            real_source_root,
        )

        def link(source, destination):
            logging.debug('Symlink from %s to %s' % (source, destination))
            if os.path.realpath(source) != os.path.realpath(destination):
                sdk_paths.remove_path(destination)
                os.symlink(source, destination)

        self.source_root = os.path.join(self.room_path, 'arcadia')
        link(real_source_root, self.source_root)


class AutocheckBuildError(errors.TaskError):
    pass


class AutocheckConfigureError(errors.TaskError):
    pass


class PessimizedByNodesError(errors.TaskError):
    pass


class KVStore(object):
    def __init__(self, **kvargs):
        self.__dict__.update(kvargs)

    def as_dict(self):
        return self.__dict__.copy()


class SubtaskStatus(object):
    SUCCESS = 0
    FAILED = 1
    STATUS_PREPARE_FAILED = 2
    PESSIMIZED = 3

    def __init__(self, partition, code):
        self.partition = partition
        self.code = code
        self.err_message = None
        self.time_metrics = None
        self.stages = None
        self.cmake_work_time = None
        self.make_work_time = None
        self.tests_work_time = None
        self.retry_partition = None

    @staticmethod
    def prepare(partition, code, context, err_message):
        status = SubtaskStatus(partition, code)
        try:
            if code == SubtaskStatus.SUCCESS:
                status.time_metrics = context.time_metrics
                status.stages = context.STAGES
                status.cmake_work_time = context.cmake_work_time
                status.make_work_time = context.make_work_time
                status.tests_work_time = context.tests_work_time
            elif code == SubtaskStatus.FAILED:
                status.err_message = err_message
                status.retry_partition = context.retry_partition
        except:
            logging.debug('Error during subtask status preparing:\n%s', traceback.format_exc())
            status = SubtaskStatus(partition, SubtaskStatus.STATUS_PREPARE_FAILED)
        return status


def load_build_statistics(filepath):
    if os.path.exists(filepath):
        with open(filepath) as f:
            return json.load(f)
    return None


class AutocheckBuildYa2Subtask(multiprocessing.Process):
    def __init__(self, partition, task, options, gsid, lock, queue):
        # type: (int, tp.Any, PartitionOptions, GSID, tp.Any, tp.Any) -> None
        super(AutocheckBuildYa2Subtask, self).__init__()
        self.name = 'Partition-{}'.format(partition)
        self.partition = partition
        self.task = task
        self.priority_calculator = self.task.priority_calculator
        self.autocheck_config = self.task.autocheck_config
        self.arc_adapter_factory = self.task.arc_adapter_factory
        self.profiler = task.profiler
        self.lock = lock
        self.options = options
        self.partition_config = self.autocheck_config[self.options.partition_index]
        self.context = KVStore(
            time_metrics={},
            is_pessimized=False,
            STAGES={},
            GSID=gsid,
            cmake_work_time=0,
            make_work_time=0,
            tests_work_time=0,
            retry_partition=True,  # by default always retry failed partitions
        )
        self.queue = queue
        self.type = 'AUTOCHECK_BUILD_YA_2'
        self.resource_data = None
        self.logs_dir = None

    def run(self):
        self.setup_logging()
        with self.profiler.start_profile_thread(title='partition_{}'.format(self.partition)):
            self._run()

    def _run(self):
        try:
            logging.info('%d: Started', self.pid)
            self.execute()
        except PessimizedByNodesError:
            logging.info('%d: check will be pessimized by nodes', self.pid)
            self.send_result('broken', SubtaskStatus.PESSIMIZED)
        except Exception as e:
            logging.exception('%d: Raised exception', self.pid)
            self.send_result('broken', SubtaskStatus.FAILED, str(e))
        else:
            logging.info('%d: Success', self.pid)
            self.send_result('ready', SubtaskStatus.SUCCESS)
        finally:
            logging.info('%d: Finished', self.pid)

    def send_result(self, res_status, partition_status, err_message=None):
        with self.lock:
            self.try_mark_resource(res_status)
            sdk_paths.remove_path(self.path())
            self.queue.put(SubtaskStatus.prepare(self.partition, partition_status, self.context, err_message))

    def try_mark_resource(self, res_status):
        try:
            if self.resource_data:
                if res_status == 'ready':
                    self.resource_data.ready()
                elif res_status == 'broken':
                    self.resource_data.broken()
        except Exception:
            logging.error('%d: Fail to mark resource %s as %s:\n%s', self.pid, self.resource_data, res_status, traceback.format_exc())
            self.resource_data.broken()
            return False
        return True

    def setup_logging(self):
        logger = logging.getLogger()

        for handler in logger.handlers:
            if isinstance(handler, logging.FileHandler):
                logger.removeHandler(handler)

        common_handler = logging.FileHandler(str(self.log_path(ctt.LogName.COMMON)))
        common_handler.setLevel(logging.INFO)
        common_handler.setFormatter(logging.Formatter(ctt.TASK_LOG_FORMAT))
        common_handler.addFilter(log.TimeDeltaMeasurer())
        logger.addHandler(common_handler)

        debug_handler = logging.FileHandler(str(self.log_path(ctt.LogName.DEBUG)))
        debug_handler.setLevel(logging.DEBUG)
        debug_handler.setFormatter(logging.Formatter(ctt.TASK_LOG_FORMAT))
        debug_handler.addFilter(log.TimeDeltaMeasurer())
        debug_handler.vault_filter = log.VaultFilter()
        debug_handler.addFilter(debug_handler.vault_filter)
        logger.addHandler(debug_handler)

    @property
    def is_autocheck_20(self):
        if self.options.is_autocheck_20:
            return True
        if self.is_precommit:
            return True
        return False

    @property
    def is_precommit(self):
        # Note: AUTOCHECK_EMULATION_TASK do not set is_precommit
        return self.options.is_precommit

    @property
    def should_be_pessimized(self):
        return self.options.should_pessimize_by_nodes

    def create_streaming_client(self, url):
        import yalibrary.streaming_client as sc

        self.ci_logbroker_partition_group = None
        if self.options.report_to_ci or (self.options.use_streaming and self.options.streaming_backends and self.options.streaming_id):
            logging.debug('Create streaming client')
            if enable_streaming_diractly_to_ci(self.options):
                if self.options.report_to_ci:
                    check_type = self.options.ci_check_type
                    if not check_type:
                        if self.options.circuit_type == 'fast':
                            check_type = "FAST"
                        else:
                            check_type = "FULL"
                    adapter = sc.StreamingCIAdapter(
                        logbroker_token=autocheck_tokens.get_logbroker_token(self.task.owner),
                        topic=self.options.ci_logbroker_topic,
                        source_id='{source_id}/{partition}/task'.format(source_id=self.options.ci_logbroker_source_id, partition=self.options.partition_index),
                        check_id=self.options.ci_check_id,
                        check_type=check_type,
                        iteration_number=self.options.ci_iteration_number,
                        partition=self.options.partition_index,
                        task_id_string=self.options.ci_task_id,
                        stream_task_id=self.task.id,
                    )
                    self.ci_logbroker_partition_group = adapter.logbroker_partition_group + 1
                    logging.debug('Use logbroker partition group: %s', self.ci_logbroker_partition_group)
                else:
                    adapter = sc.StreamingHTTPAdapter(self.options.streaming_backends[0], self.options.streaming_id or 'fake_stream_id', self.options.partition_index, self.task.id)
            else:
                adapter = sc.StreamingHTTPAdapter(url, self.options.streaming_id or 'fake_stream_id', self.options.partition_index, self.task.id)
            stream_client = sc.StreamingClient(adapter, self.task.id)
            return stream_client
        logging.warning('Skip creating streaming client')
        return None

    def execute(self):
        import contextlib2

        logging.debug('Task parameters are %s', json.dumps(self.options.as_dict(), indent=4, sort_keys=True))
        logging.debug('Task context is %s', json.dumps(self.context.as_dict(), indent=4, sort_keys=True))

        if self.options.custom_recheck and not self.options.custom_targets_list:
            logging.debug('No targets for recheck')
            return

        os.environ['TMPDIR'] = str(self.path('_tmp'))
        sdk_paths.make_folder(self.path('_tmp'), delete_content=True)

        env = self._make_env()
        ymake.set_custom_fetcher(env)

        build_state = BuildState(self.options)
        logs_resource = self.create_autocheck_logs_resource('build')
        self.start_streaming_server(build_state)
        self.stream_client = self.create_streaming_client(build_state.report_server_url)

        self.arc_thin_adapter_factory = self.arc_adapter_factory.make_arc_thin_adapter_factory()
        with contextlib2.ExitStack() as stack:
            stack.enter_context(self.arc_thin_adapter_factory)
            stack.enter_context(process.CustomOsEnviron(env))
            try:
                self._do_execute(build_state, logs_resource)
            except Exception:
                task_error_message = traceback.format_exc()
                err = autocheck_errors.is_none_retriable(task_error_message)
                if self.is_precommit and err:
                    logging.info('Partition invoked non retriable error:\n{}'.format(err.message))
                    self.context.retry_partition = False
                    if self.stream_client:
                        self.stream_client.send_autocheck_fatal_error(err.summary, err.message)
                raise
            finally:
                if build_state.report_server is not None:
                    build_state.report_server.stop()
                    for r in build_state.receivers:
                        r.close()

    def stage(self, name, t=None, sync=False):
        stage_time = t or time.time()
        self.context.STAGES[name] = stage_time
        if sync:
            json.dump(self.context.STAGES, open(str(self.path('stages.json')), 'wt'), indent=2)

    def stage_started(self, name, t=None, sync=False):
        self.stage(name + '_started', t=t, sync=sync)

    def stage_finished(self, name, t=None, sync=False):
        self.stage(name + '_finished', t=t, sync=sync)

    def load_stages(self, stages_path):
        if os.path.exists(stages_path):
            stages = json.load(open(stages_path))
            for k, v in sorted(stages.items(), key=lambda x: x[1]):
                self.stage(k, v, sync=False)

    def update_time_metrics(self, name, value, sync=False):
        self.context.time_metrics[name] = value
        if sync:
            json.dumps(self.context.time_metrics, open(str(self.path('time_metrics.json')), 'wt'), indent=2)

    def create_autocheck_logs_resource(self, name_prefix, ttl=None):
        revision = self.options.autocheck_revision
        ttl = ttl or self.options.autocheck_logs_resource_ttl

        with self.lock:
            logs_dir = '{prefix}ya_check_{partition}_{count}'.format(
                prefix=name_prefix,
                partition=self.partition,
                count=len(list(resources.AutocheckLogs.find(task=self.task, attrs={'partition': self.partition}).limit(1000))) + 1
            )
            resource = resources.AutocheckLogs(
                self.task, '%s r%d partition %d autocheck logs' % (name_prefix, revision, self.partition), logs_dir, ttl=ttl
            )
            resource.partition = self.partition
            resource.revision = revision
            self.resource_data = sdk2.ResourceData(resource)
            self.resource_data.path.mkdir()
            self.logs_dir = str(self.resource_data.path.absolute())
            return resource

    def _make_env(self):
        data = {
            'YA_STRIP_PACKAGES_FROM_RESULTS': '1',
            'YA_NO_GEN_RENAMED_RESULTS': '1',
            'YA_GRAPH_GEN_DIST_PRIORITY': str(self.options.normalized_dist_priority),
            'TOOLCHAIN': self.options.autocheck_toolchain_new,
            'YA_NO_RESPAWN': '1',
            'AUTOCHECK': '1',
        }
        return data

    def path(self, *args):
        task_dir = self.task.path(self.partition)
        if not task_dir.exists():
            task_dir.mkdir()
        return task_dir.joinpath(*args)

    def log_path(self, *args):
        log_dir = self.task.log_path(self.partition)
        if not log_dir.exists():
            log_dir.mkdir()
        return log_dir.joinpath(*args)

    def _do_execute(self, build_state, logs_resource):
        self.task_start_time = time.time()

        sdk_paths.remove_path(self.path('exported_arcadia'))
        repo = self.arc_thin_adapter_factory.make_arc_adapter(str(self.path('exported_arcadia')), svn.Arcadia.normalize_url(self.options.arcadia_url), self.options.arc_patch)

        self.append_to_gsid('PARTITION', self.options.partition_index)
        if self.options.native_build_targets:
            try:
                build_state.native_builds = native_builds.NativeBuilds(self.options.native_build_targets)
            except ValueError:
                logging.error('Cannot load params for native builds: %s', self.options.native_build_targets)

        if self.options.send_logs_to_logbroker:
            self.un_agent = unified_agent_wrapper.unified_agent_wrapper()

        self.trace_stage('started')

        paths, changed_options, cleanup_callback = self.build_graphs_intersection(build_state, repo)

        if not self.options.stop_after_meta_graph:
            if not self.is_autocheck_20 or self.options.custom_recheck:
                build_mode = BuildMode.GPAPH_AND_BUILD
            else:
                build_mode = BuildMode.ONLY_BUILD

            self.build(logs_resource, changed_options, build_state, repo, paths, build_mode)

        self.trace_stage('finished')

        if self.options.send_logs_to_logbroker:
            self.un_agent.close()

        cleanup_callback()

    def prepare_meta_graph_resource(self,
                                    options,  # type: PartitionOptions
                                    left_working_copy,  # tuple
                                    right_working_copy,  # tuple
                                    ):
        logging.debug("Prepare resource for store meta graphs")
        assert not hasattr(self, '_store_meta_graph_data'), "prepare_meta_graph_resource was already called"

        with self.lock:
            autocheck_config_path = options.config_path
            targets = ';'.join(options.targets)
            partition_index = self.partition
            partitions_count = self.options.partitions_count
            left_rev, left_url, left_patch = left_working_copy
            right_rev, right_url, right_patch = right_working_copy
            existed_resources_count = len(list(resources.YaMakeMetaGraphResult.find(task=self.task, attrs={'partition_index': self.partition}).limit(1000))) + 1
            if self.options.acceptance_task:
                kind = 'acceptance'
            else:
                # TODO: Use binary_executor_release_type from ctx
                kind = str(self.task.Parameters.binary_executor_release_type)

            source = str(self.task.type)

            desc = 'Meta Graphs {} ({}) {} ({} {} {}) ({} {} {}) {} {}/{} #{}'.format(
                source,
                kind,
                self.options.circuit_type,
                left_rev, left_url, left_patch,
                right_rev, right_url, right_patch,
                targets,
                partition_index, partitions_count,
                existed_resources_count,
            )

            logging.debug("Resource description for store meta graphs: %s",
                          desc)

            resource = resources.YaMakeMetaGraphResult(
                task=self.task,
                description=desc,
                path='meta_graphs_{}_{}_{}'.format(partition_index, partitions_count, existed_resources_count),
                left_revision=left_rev,
                left_url=left_url,
                left_patch=left_patch,
                right_revision=right_rev,
                right_url=right_url,
                right_patch=right_patch,
                circuit_type=self.options.circuit_type,
                is_dummy_arcadia=targets == 'devtools/dummy_arcadia/empty_project',
                targets=targets,
                autocheck_config_path=autocheck_config_path,
                partition_index=partition_index,
                partitions_count=partitions_count,
                kind=kind,
                source=source,
                correct_order=True,
            )

            logging.info("Resource id for store %s: %s", desc, resource.id)

            self._store_meta_graph_data = sdk2.ResourceData(resource)
            self._store_meta_graph_data.path.mkdir(mode=0o755, parents=True, exist_ok=False)

    def store_meta_graph_resource(self, graph_type, graph_path):
        assert hasattr(self, '_store_meta_graph_data'), "call `prepare_meta_graph_resource` first"

        assert graph_type in ['left', 'right', 'subtract']
        assert os.path.exists(graph_path)

        partition_index = self.options.partition_index
        partitions_count = self.options.partitions_count

        folder_to_store = str(self._store_meta_graph_data.path)

        path_to_store = os.path.join(folder_to_store, "{}.json".format(graph_type))
        assert not os.path.exists(path_to_store)

        logging.debug("Store `%s` meta graph from partition %d/%d into %s",
                      graph_type,
                      partition_index, partitions_count,
                      path_to_store)

        with open(graph_path, 'rt') as source:
            with open(path_to_store, "wt") as dest:
                dest.write(source.read())

        logging.info("Meta graph `%s` from partition %d/%d dumped from %s to %s",
                     graph_type,
                     partition_index, partitions_count,
                     graph_path, path_to_store)

    def finish_meta_graph_resource(self):
        logging.debug("Check resource for store meta graphs")
        assert hasattr(self, '_store_meta_graph_data'), "prepare_meta_graph_resource was already called"

        with self.lock:
            files_count = len(os.listdir(str(self._store_meta_graph_data.path)))
            if files_count != 3:
                logging.warning("Not enough items in resource folder. Expect 3, but in fact %d", files_count)
                self._store_meta_graph_data.broken()
            else:
                logging.info("Folders with meta graphs are complete")
                self._store_meta_graph_data.ready()

    def build_graphs_intersection(self, build_state, repo):
        if not self.is_autocheck_20 or self.options.custom_recheck:
            paths = self.prepare_for_build_graph(self.options, repo)

            def cleanup_callback():
                if self.options.keep_all_logs_and_files:
                    self.tar_temporary_logs(paths, None)

            return paths, self.options, cleanup_callback

        comparing_graph_build_state = None
        comparing_graph_paths = None

        self.trace_stage('graph/started')
        if self.options.use_in_compare_repo_state:
            comparing_repo_options = self.options.get_comparing_repo_options()
            sdk_paths.remove_path(self.path('comparing_exported_arcadia'))
            comparing_repo = self.arc_thin_adapter_factory.make_arc_adapter(
                str(self.path('comparing_exported_arcadia')),
                svn.Arcadia.normalize_url(comparing_repo_options.arcadia_url),
                comparing_repo_options.arc_patch
            )
            comparing_graph_paths = self.prepare_for_build_graph(
                comparing_repo_options, comparing_repo, prefix='comparing_repos_state_graph_build'
            )

        paths = self.prepare_for_build_graph(self.options, repo, prefix='graph_build')

        if self.options.circuit_type == 'fast':
            from devtools.platform_miner import miner
            from yalibrary import platform_matcher
            if self.options.arcadia_patch:
                arcadia_root = comparing_graph_paths.source_root
            else:
                arcadia_root = paths.source_root
            ya_bin = ymake.mine_ya(arcadia_root)
            ya_path = os.path.join(*ya_bin)
            autocheck_config_fullpath = os.path.join(arcadia_root, self.options.config_path)
            mined_targets = miner.mine_platforms(ya_path, arcadia_root, autocheck_config_fullpath, self.options.targets,
                                                 self.options.partition_index, self.options.partitions_count)
            logging.debug('Mined targets for check: %s', mined_targets)
            if mined_targets:
                processed_mined_targets = []
                for path, toolchain_string, project_index in mined_targets:
                    alias = platform_matcher.get_target_platform_alias(toolchain_string, self.options.toolchain_transforms)
                    eh.ensure(alias, 'Fail fast circuit. Unable to find alias for platform {}'.format(toolchain_string))
                    processed_mined_targets.append([path, alias, project_index])
                self.options.set_custom_targets_list(processed_mined_targets, False)
                if self.options.use_in_compare_repo_state:
                    comparing_repo_options.set_custom_targets_list(processed_mined_targets, False)
            else:
                # build stub project in case there is no mined targets
                # this is cheap way to emulate check and push messages to stream processor
                targets = ['devtools/dummy_arcadia/empty_project']
                target_platforms = self.options.target_platforms
                self.options.targets = targets
                self.options.target_platforms = target_platforms
                if self.options.use_in_compare_repo_state:
                    comparing_repo_options.targets = targets
                    comparing_repo_options.target_platforms = target_platforms

        try:
            if self.options.save_meta_graphs:
                try:
                    right_working_copy = (repo.revision, self.options.checkout_arcadia_from_url, self.options.arc_patch)
                    if self.options.use_in_compare_repo_state:
                        left_working_copy = (comparing_repo.revision, self.options.checkout_arcadia_from_url_for_in_compare_repo_state, comparing_repo_options.arc_patch)
                    else:
                        left_working_copy = (None, None, None)
                    self.prepare_meta_graph_resource(self.options, left_working_copy, right_working_copy)
                except Exception:
                    logging.exception("While preparing meta graphs resource something goes wrong, disabling save_meta_graphs")
                    self.options.save_meta_graphs = False

            else:
                logging.debug("Preparing meta graph disabled because save_meta_graphs:%s",
                              self.options.save_meta_graphs)

            if self.options.use_in_compare_repo_state:
                comparing_graph_build_state = BuildState(comparing_repo_options)
                self.build_graph(comparing_repo, comparing_repo_options, 'comparing_repos_state_graph_build', comparing_graph_build_state)

            self.build_graph(repo, self.options, 'graph_build', build_state)

            subtract_on_dist_resource_path = os.path.join(self.logs_dir, 'subtract_on_dist')

            context_path = self.subtract_intersecting_between_graphs_and_contexts_on_dist(
                comparing_graph_build_state, build_state,
                paths, subtract_on_dist_resource_path,
                self.options.recheck,
                self.options.normalized_dist_priority,
                self.options.cache_namespace,
                self.options.save_meta_graphs,
                self.options.gg_coordinators_filter,
                self.options.gg_distbs_pool,
                repo,
            )
            self.trace_stage('graph/finished')
        except:
            self.tar_temporary_logs(paths, comparing_graph_paths)
            raise
        finally:
            if self.options.save_meta_graphs:
                try:
                    self.finish_meta_graph_resource()
                except Exception:
                    logging.exception("While finish meta graphs resource")
            else:
                logging.debug("Finishing meta graph disabled because save_meta_graphs:%s",
                              self.options.save_meta_graphs)

        build_state.context_path = context_path
        self.send_estimated_cpu_time(build_state.context_path, build_state.uids_for_affected_nodes)
        changed_options = copy.deepcopy(self.options)
        spec_gsid_parts = []
        if build_state.uids_for_affected_nodes is not None and build_state.uids is not None:
            logging.info('Affected nodes count = %s', len(build_state.uids_for_affected_nodes))
            logging.info('Graph nodes count = %s', len(build_state.uids))
            if self.stream_client:
                self.stream_client.send_number_of_nodes(len(build_state.uids_for_affected_nodes))

            if self.is_precommit:
                # Lower pessimization by nodes threashold for common_components pool
                if not self.should_be_pessimized and len(build_state.uids_for_affected_nodes) > LOW_CACHE_HIT_GRAPH_THRESHOLD:
                    logging.info('Pessimize check: partition exceeds limit by nodes: %s', LOW_CACHE_HIT_GRAPH_THRESHOLD)
                    if self.stream_client:
                        self.stream_client.send_pessimize('Check is pessimized due to the large number of affected nodes')
                    yt_report.send_pessimized_by_nodes(
                        yt_report.TaskInfoReport.create_task(self), len(build_state.uids_for_affected_nodes), LOW_CACHE_HIT_GRAPH_THRESHOLD, self.un_agent)
                    raise PessimizedByNodesError()

                if len(build_state.uids_for_affected_nodes) > LOW_CACHE_HIT_GRAPH_THRESHOLD:
                    self.context.is_pessimized = True
                    spec_gsid_parts += ['SB:{}:{}:LOW_CACHE_HIT'.format(self.type, len(build_state.uids_for_affected_nodes))]
                    changed_options.normalized_dist_priority = self.priority_calculator.low_cache_hit
                elif len(build_state.uids_for_affected_nodes) > MEDIUM_CACHE_HIT_GRAPH_THRESHOLD:
                    self.context.is_pessimized = True
                    spec_gsid_parts += ['SB:{}:{}:MEDIUM_CACHE_HIT'.format(self.type, len(build_state.uids_for_affected_nodes))]
                    changed_options.normalized_dist_priority = self.priority_calculator.medium_cache_hit

            if len(build_state.uids) > LOW_CACHE_HIT_GRAPH_THRESHOLD:
                spec_gsid_parts += ['SB:{}:{}:BIG_GRAPH'.format(self.type, len(build_state.uids))]
            elif len(build_state.uids) > MEDIUM_CACHE_HIT_GRAPH_THRESHOLD:
                spec_gsid_parts += ['SB:{}:{}:MEDIUM_GRAPH'.format(self.type, len(build_state.uids))]

        if self.should_be_pessimized:
            self.context.is_pessimized = True
            spec_gsid_parts += ['SB:{}:PESSIMIZED_BY_PARENT'.format(self.type)]
            changed_options.normalized_dist_priority = self.priority_calculator.low_cache_hit

        if not self.options.is_trunk:
            self.context.is_pessimized = True
            spec_gsid_parts += ['SB:{}:PESSIMIZED_BRANCH'.format(self.type)]
            changed_options.normalized_dist_priority = self.priority_calculator.low_cache_hit

        # pessimize ya check / ya shelve calls
        if 'YA:USE_WORKING_COPY_REVISION' in self.context.GSID:
            self.context.is_pessimized = True
            changed_options.normalized_dist_priority = self.priority_calculator.low_cache_hit

        if self.context.is_pessimized:
            spec_gsid_parts += ['SB:PESSIMIZED']

        # this GSID modification used by distbuild to classify this build properly
        for spec_gsid_part in spec_gsid_parts:
            self.append_to_gsid(spec_gsid_part)

        def cleanup_callback():
            if self.options.keep_all_logs_and_files:
                self.tar_temporary_logs(paths, comparing_graph_paths)

            self._tar_and_clean(os.path.dirname(os.path.dirname(build_state.meta_graph_result.result_dir)))
            if self.options.use_in_compare_repo_state:
                self._tar_and_clean(os.path.dirname(os.path.dirname(comparing_graph_build_state.meta_graph_result.result_dir)))

            self._tar_and_clean(subtract_on_dist_resource_path)

        return paths, changed_options, cleanup_callback

    def append_to_gsid(self, name, value=None):
        if name not in self.context.GSID:
            gsid_value = name
            if value is not None:
                gsid_value += ':' + str(value)
            self.context.GSID += ' ' + gsid_value

    def generate_subtract_meta_graph(self, left_meta_graph_result, right_meta_graph_result, repo, result_dir, recheck, native_builds):
        from yalibrary.ya_helper import common
        from yalibrary.ggaas.builders import subtract_meta_graph
        from yalibrary.ggaas.interface import build

        subtract_request = build.SubtractBuildForPlatform(
            configuration=self.autocheck_config,
            partition_index=self.options.partition_index,
            partitions_count=self.options.partitions_count,
            generate_left_right=self.options.build_left_right_graph,
            recheck=recheck and self.is_autocheck_20,
            native_build_targets=native_builds,
            download_left_graph=self.options.save_graph_and_context,
            download_right_graph=self.options.save_graph_and_context,
        )

        subtract_generator = subtract_meta_graph.MetaGraphSubtractGenerator(
            left_graph_result=left_meta_graph_result,
            right_graph_result=right_meta_graph_result,
            build_request=subtract_request,
            arc_repo=repo,
            result_dir=result_dir,
        )

        try:
            return subtract_generator.generate_graph_subtract_calculation_graph()
        except common.SubprocessError as e:
            raise AutocheckConfigureError(e.stderr)

    def execute_subtract_meta_graph(self, subtract_result, repo, result_dir, cache_namespace, coordinators_filter, distbuild_pool):
        # type: (tp.Any, tp.Any, tp.Any, tp.Any, tp.Any, tp.Any) -> GraphsSubtractExecutionResult
        from yalibrary.ya_helper import common
        from yalibrary.ggaas.builders import execute_graph
        from yalibrary.ggaas.interface import execute

        if self.options.circuit_type == 'fast':
            build_profile = execute.ExecuteOptions.BuildProfile.AUTOCHECK_FAST_PRE_COMMIT
        else:
            build_profile = execute.ExecuteOptions.BuildProfile.AUTOCHECK_PRE_COMMIT

        execute_options = execute.ExecuteOptions(
            ya_custom_options=execute.YaCustomExecuteOptions(
                cache_namespace=cache_namespace,
                coordinators_filter=coordinators_filter,
                distbs_pool=distbuild_pool,
                distbs_timeout=self.options.distbs_timeout,
                distbuild_testing_cluster_id=self.options.distbuild_testing_cluster_id,
                use_distbuild_testing_cluster=self.options.use_distbuild_testing_cluster,
            ),
            build_profile=build_profile,
            gsid_parts=common.gsid.GSIDParts(self.context.GSID),
            priority_base=self.options.autocheck_revision,
        )

        executor = execute_graph.MetaGraphExecutor(
            graph=subtract_result,
            execute_options=execute_options,
            arc_repo=repo,
            result_dir=result_dir
        )

        from yalibrary.ggaas.interface.execute import GraphsSubtractExecutionResult  # noqa
        try:
            return executor.execute()  # type: GraphsSubtractExecutionResult
        except common.SubprocessError as e:
            raise AutocheckConfigureError(e.stderr)

    def subtract_intersecting_between_graphs_and_contexts_on_dist(
        self, comparing_graph_build_state, build_state, paths, logs_resource_path,
        recheck, dist_priority, cache_namespace,
        save_meta_graphs, graph_coordinators_filter, graph_distbuild_pool,
        repo,
    ):
        from yalibrary.ggaas.builders.execute_graph import NotRetryableGraphGenerationError, FallbackType
        ya_bin = ymake.mine_ya(paths.source_root)
        have_fallback = build_state.have_fallback_meta_graph and comparing_graph_build_state.have_fallback_meta_graph
        have_minimal_cache = build_state.have_minimal_cache_meta_graph and comparing_graph_build_state.have_minimal_cache_meta_graph

        native_builds = build_state.native_builds.combine_targets() if self.options.native_build_targets else None

        subtract_result = self.generate_subtract_meta_graph(
            comparing_graph_build_state.meta_graph_result,
            build_state.meta_graph_result,
            repo, os.path.join(logs_resource_path, 'context_with_cache'), recheck, native_builds
        )

        if have_minimal_cache:
            subtract_result_minimal_cache = self.generate_subtract_meta_graph(
                comparing_graph_build_state.meta_graph_result.minimal_cache,
                build_state.meta_graph_result.minimal_cache,
                repo, os.path.join(logs_resource_path, 'context_with_minimal_cache'), recheck, native_builds
            )

        if have_fallback:
            subtract_result_fallback = self.generate_subtract_meta_graph(
                comparing_graph_build_state.meta_graph_result.fallback,
                build_state.meta_graph_result.fallback,
                repo, os.path.join(logs_resource_path, 'context_without_cache'), recheck, native_builds
            )

        if save_meta_graphs:
            try:
                graph_type = 'subtract'
                self.store_meta_graph_resource(graph_type, subtract_result.subtracting_context_result)
            except Exception:
                logging.exception("While dumping %s meta graph", graph_type)
        else:
            logging.debug("Dumping meta graph disabled because save_meta_graphs:%s",
                          save_meta_graphs)

        self.stage_started('sb-subtract-intersecting-contexts')

        def execute_meta_graph(graph_result, result_sub_dir):
            result_path = os.path.join(logs_resource_path, result_sub_dir)
            return self.execute_subtract_meta_graph(graph_result, repo, result_path, cache_namespace, graph_coordinators_filter, graph_distbuild_pool)

        try:
            try:
                execute_result = execute_meta_graph(subtract_result, 'execute')
                cache_mode = yt_report.CacheMode.FULL
            except NotRetryableGraphGenerationError as e:
                if not (have_minimal_cache and e.fallback_type == FallbackType.DOWNGRADE_CACHE):
                    raise
                logging.exception('Failed to execute subtracting meta-context on distbuild; switching to the minimal cache mode.')
                execute_result = execute_meta_graph(subtract_result_minimal_cache, 'execute_minimal_cache')
                cache_mode = yt_report.CacheMode.MINIMAL
        except NotRetryableGraphGenerationError as e:
            cache_mode = yt_report.CacheMode.NONE
            logging.exception('Failed to execute subtracting meta-context on distbuild; switching to the fallback mode.')
            execute_result = execute_meta_graph(subtract_result_fallback, 'execute_fallback')

        # Give result subdirs better/readable names
        for state, side in [(comparing_graph_build_state.meta_graph_result, 'left'), (build_state.meta_graph_result, 'right')]:
            os.rename(os.path.join(execute_result.result_dir, state.remote_build_dir), os.path.join(execute_result.result_dir, side))
            # Also unpack generating.logs for humans
            generating_log_uc = execute_result.generating_log(side)
            if os.path.exists(generating_log_uc):
                generating_log = generating_log_uc[:-3]
                with sdk2.helpers.ProcessLog(self.task, logger=logging.getLogger('ya_tool_uc')) as pl:
                    proc = sdk2.helpers.subprocess.run(
                        ya_bin + ['--no-report', 'tool', 'uc', '-d', '-f', generating_log_uc, '-t', generating_log],
                        stdout=pl.stdout,
                        stderr=sdk2.helpers.subprocess.STDOUT)
                if proc.returncode == 0:
                    sdk_paths.remove_path(generating_log_uc)
                else:
                    logging.warning('Unable to unpack "%s"', generating_log_uc)

        if self.options.send_logs_to_logbroker:
            yt_report.send_gg_cache_stats(
                yt_report.TaskInfoReport.create_task(self),
                execute_result.result_dir,
                self.un_agent,
                cache_mode=cache_mode,
                gg_cache_is_found=have_fallback,
                autocheck_config_path=self.task.Context.autocheck_config_path,
                circuit_type=self.task.Parameters.circuit_type,
                is_precommit=self.task.Parameters.is_precommit,
                recheck=self.task.Parameters.recheck,
            )

        try:
            uids_for_affected_nodes = execute_result.affected_uids_right_left
            with open(uids_for_affected_nodes) as f:
                build_state.uids_for_affected_nodes = set(json.load(f))
        except Exception as e:
            logging.info('Failed to get uids_for_affected_nodes, path is %s, error is %s', uids_for_affected_nodes, repr(e))

        remaining_uids = execute_result.remaining_uids
        try:
            with open(remaining_uids) as f:
                build_state.uids = set(json.load(f))
        except Exception as e:
            logging.info('Failed to get remaining_uids, path is %s, error is %s', remaining_uids, repr(e))

        if self.options.native_build_targets:
            build_state.native_builds.set_bin_result_path(execute_result.bin_result_paths)

        # Repeat the same we do at the end of ONLY_GRAPH mode: modify stages, calc build_graph_last_stage and update metrics.
        # But we do not have the same profile.json and stages.json. Let's generate them first in the similar way.
        def join_profiles(main_profile, sub_profile):
            prefix = 'cg_'
            main_start_time = main_profile.get('step', {}).get('start_time')
            sub_start_time = sub_profile.get('step', {}).get('start_time')
            delta = sub_start_time - main_start_time if main_start_time and sub_start_time else 0
            for namespace, data in six.iteritems(sub_profile):
                if namespace != 'step':
                    for name, value in six.iteritems(data):
                        main_profile[namespace][prefix + name] = value
            for name, value in six.iteritems(sub_profile.get('step', {})):
                main_profile['step'][prefix + name] = value + delta

        def join_stages(main_stages, sub_stages):
            prefix = 'cg_'
            for k, v in six.iteritems(sub_stages):
                main_stages[prefix + k] = v

        def load_json(profile_path):
            try:
                return json.load(open(profile_path))
            except:
                return {}

        main_profile = load_json(execute_result.profile)
        main_stages = load_json(execute_result.stages)
        for state, side in [(comparing_graph_build_state, 'left'), (build_state, 'right')]:
            new_profile = copy.deepcopy(main_profile)
            side_profile = load_json(execute_result.context_profile(side))
            join_profiles(new_profile, side_profile)

            state.build_graph_profile_path = execute_result.full_profile(side)
            with open(state.build_graph_profile_path, 'w') as f:
                json.dump(new_profile, f, indent=4, sort_keys=True)

            new_stages = copy.deepcopy(main_stages)
            side_stages = load_json(execute_result.context_stages(side))
            join_stages(new_stages, side_stages)

            state.build_graph_stages_path = execute_result.full_stages(side)
            with open(state.build_graph_stages_path, 'w') as f:
                json.dump(new_stages, f, indent=4, sort_keys=True)
            state.build_graph_stages_path, stages = self.modify_stages(state.start_time, '', state.build_graph_stages_path)
            state.build_graph_last_stage = max(stages.values()) if stages else None
        self.load_graph_stages_and_profile(comparing_graph_build_state, build_state)

        if self.options.send_logs_to_logbroker:
            yt_report.send_distbs_overhead(yt_report.TaskInfoReport.create_task(self), execute_result.result_dir, self.un_agent, graph_gen=True)

        self.stage_finished('sb-subtract-intersecting-contexts')
        substracted_graph_dir = 'left-right' if self.options.build_left_right_graph else 'right-left'
        return os.path.join(execute_result.result_dir, substracted_graph_dir, 'context_cut.json.uc')

    def prepare_source_roots(self, arcadia_patch, repo, prefix=None):
        log_prefix = '{}: '.format(prefix) if prefix is not None else ''
        name_prefix = '{}_'.format(prefix) if prefix is not None else ''

        logging.debug('%sRepo is %s, url is %s', log_prefix, repo.__dict__, repo.get_arcadia_url())

        logging.debug('%sGetting source root', log_prefix)
        start_access_repo_time = time.time()
        self.stage_started(name_prefix + 'svn-source')
        source_root = repo.get_source_root()
        self.stage_finished(name_prefix + 'svn-source')
        self.commit_time = repo.get_commit_time()
        self.update_time_metrics(name_prefix + 'svn_source_finished', time.time() - self.commit_time)
        self.update_time_metrics(name_prefix + 'svn_source_started', start_access_repo_time - self.commit_time)

        paths = Paths(str(self.path(name_prefix + '_work_dir')))
        paths.setup(source_root)

        logging.debug('%sPaths are %s', log_prefix, json.dumps(paths.__dict__, indent=4, sort_keys=True))

        self.update_time_metrics('commit_time', self.commit_time)
        self.update_time_metrics('task_started', self.task_start_time - self.commit_time)
        self.update_time_metrics('task_created', calendar.timegm(self.task.created.timetuple()) - self.commit_time)

        return paths

    def prepare_for_build_graph(self, options, repo, prefix=None):
        log_prefix = '{}: '.format(prefix) if prefix is not None else ''

        paths = self.prepare_source_roots(options.arc_patch, repo, prefix)

        options.load_config(paths.source_root)
        logging.debug('%sOptions are %s', log_prefix, options.__dict__)

        return paths

    def build_graph(self, repo, options, prefix, build_state):
        from yalibrary.ggaas.builders import meta_graph
        from yalibrary.ya_helper.common import gsid
        from yalibrary.ggaas.interface import build, graph_request

        build_state.start_time = time.time()

        # postcommits and precommits are using the same profile
        if self.options.circuit_type == 'fast':
            build_profile = build.BuildOptions.BuildProfile.AUTOCHECK_FAST_PRE_COMMIT
        else:
            build_profile = build.BuildOptions.BuildProfile.AUTOCHECK_PRE_COMMIT

        build_options = build.BuildOptions(
            ya_custom_options=build.YaCustomBuildOptions(
                distbs_pool=options.gg_distbs_pool,
                coordinators_filter=options.gg_coordinators_filter,
                streaming_task_id=self.task.id,
                custom_targets_list=options.custom_targets_list,
                toolchain_transforms=options.toolchain_transforms,
            ),
            build_profile=build_profile,
            graph_name=prefix,
            is_comparing_repo=options.is_comparing_repo,
            gsid_parts=gsid.GSIDParts(self.context.GSID),
            graph_distbs_pool=options.gg_distbs_pool,
            graph_coordinators_filter=options.gg_coordinators_filter,
            merge_split_tests=self.options.merge_split_tests,
            high_priority_mds_read=self.is_precommit and not self.context.is_pessimized,
            ymake_cache_kind=self.options.ymake_cache_kind,
            use_ymake_cache=self.options.use_ymake_cache,
            use_ymake_minimal_cache=self.options.used_ymake_minimal_cache_kind,
            use_imprint_cache=self.options.use_imprint_cache,
            trust_cache_fs=self.options.trust_cache_fs,
            priority_base=int(self.options.autocheck_revision),
        )

        request = graph_request.GraphBuildRequest(
            arc_repo=repo,
            build_requests=build.BuildForPlatform(
                configuration=self.autocheck_config,
                partition_index=options.partition_index,
                partitions_count=options.partitions_count,
            ),
            targets=build.Targets(*options.targets),
            build_options=build_options,
        )

        mgg = meta_graph.MetaGraphGenerator(
            graph_request=request,
            ya_bin=None,
            result_dir=os.path.join(self.logs_dir, 'ggaas_' + prefix),
        )

        # raises ValueError
        result = mgg.generate_graph_generation_graph()
        build_state.meta_graph_result = result

        self.stage_started('{}-finalization'.format(prefix))
        if options.save_meta_graphs:
            graph_type = 'left' if options.is_comparing_repo else 'right'
            try:
                self.store_meta_graph_resource(graph_type, result.meta_graph_path)
            except Exception:
                logging.exception("While dumping %s meta graph", graph_type)
        else:
            logging.debug("Dumping meta graph disabled because save_meta_graphs:%s", options.save_meta_graphs)
        misc_result_dir = os.path.dirname(result.result_dir)
        original_stages_path = os.path.join(misc_result_dir, 'stages.json')
        _, stages = self.modify_stages(build_state.start_time, '', original_stages_path)
        self.stage_finished('{}-finalization'.format(prefix))
        self.stage('last-ya-make-stage-finished-for-{}'.format(prefix), max(stages.values()) if stages else None)

    def build(self, logs_resource, options, build_state, repo, paths, build_mode):
        if self.options.check_yp_hosts and self.is_autocheck_20:
            self.append_to_gsid('EXP', 'yp_only')

        if self.options.check_yp_hosts and self.is_autocheck_20:
            coordinators_filter = options.coordinators_filter
            graph_distbs_pool = options.distbs_pool
        else:
            coordinators_filter = options.gg_coordinators_filter
            graph_distbs_pool = options.gg_distbs_pool
        custom_env = self._make_custom_env_for_build(
            paths,
            graph_coordinators_filter=coordinators_filter,
            graph_distbs_pool=graph_distbs_pool,
        )
        # TODO(workfork): replace with cli argument after 31.01.2020 (to keep backward compatibility for branches)
        custom_env['YA_STREAMING_STAGE_NAMESPACE'] = 'main'

        if self.options.native_build_targets:
            build_state.native_builds.run_native_builds(self.stream_client, repo.revision, self.is_precommit)

        build_state.start_time = time.time()
        with process.CustomOsEnviron(custom_env):
            self.trace_stage('main/started')
            builder = self.run_build(
                self.logs_dir, options, build_state, paths, repo, build_mode=build_mode
            )
            self.trace_stage('main/finished')

        # XXX: do not remove
        self.update_time_metrics('run_build_finished', time.time() - self.commit_time, sync=True)

        sdk_paths.copy_path(str(self.path('time_metrics.json')), os.path.join(self.logs_dir, 'time_metrics.json'))
        sdk_paths.remove_path(str(self.path('time_metrics.json')))

        for file_name in os.listdir(str(self.log_path())):
            sdk_paths.copy_path(str(self.log_path(file_name)), os.path.join(self.logs_dir, file_name))

        self.send_cache_hit()
        self.send_critical_path()
        self.send_build_and_test_slot_time()

        if self.options.send_logs_to_logbroker:
            yt_report.send_distbs_overhead(yt_report.TaskInfoReport.create_task(self), self.logs_dir, self.un_agent)

        resource_packager.pack_resources(self.logs_dir)

        builder.cleanup()
        self.stage_finished('build-task-finalization', sync=True)

        # XXX: do not remove
        self.update_time_metrics('build_task_finished', time.time() - self.commit_time, sync=True)
        self.update_time_metrics('task_finished', time.time() - self.commit_time, sync=True)

        if self.path('stages.json').exists():
            sdk_paths.copy_path(str(self.path('stages.json')), os.path.join(self.logs_dir, 'stages.json'))

        if self.options.send_logs_to_logbroker:
            yt_report.send_stages(yt_report.TaskInfoReport.create_task(self), self.logs_dir, self.un_agent)

    def _make_custom_env_for_build(
        self,
        paths,
        name_prefix='',
        add_make_context_on_distbuild_only=False,
        graph_coordinators_filter=None,
        graph_distbs_pool=None,
    ):
        # TODO: Use YaOptions instead
        custom_env = {
            'GSID': self.context.GSID,
            'YA_CACHE_DIR_TOOLS': os.path.join(os.environ['YA_CACHE_DIR'], 'tools'),
            'YA_CACHE_DIR': os.path.join(os.path.dirname(paths.room_path), '_ya_cache_dir'),
            'YA_EVLOG_FILE': os.path.join(self.logs_dir, name_prefix + 'event_log_file.json'),
            'YA_LOG_FILE': os.path.join(self.logs_dir, name_prefix + 'log_file.txt'),
            'YA_USE_ARCC_IN_DISTBUILD': '1',
            'YA_USE_NEW_DISTBUILD_CLIENT': '1',
        }
        if add_make_context_on_distbuild_only:
            custom_env['YA_MAKE_CONTEXT_ON_DISTBUILD_ONLY'] = '1'
        if self.is_precommit and not self.context.is_pessimized:
            custom_env['YA_HIGH_PRIORITY_MDS_READ'] = '1'
        if graph_coordinators_filter:
            custom_env['YA_GRAPH_COORDINATORS_FILTER'] = graph_coordinators_filter
        if graph_distbs_pool:
            custom_env['YA_GRAPH_DISTBUILD_POOL'] = graph_distbs_pool

        # Incremental enabling, to remove in the future
        if self.options.use_ymake_cache:
            arc_token = autocheck_tokens.get_arc_token(self.options.arc_token, self.task.owner)
            if arc_token:
                custom_env['ARC_TOKEN'] = arc_token

        ya_bin_token = autocheck_tokens.get_ya_bin_token(self.options.ya_bin_token, self.task.owner)
        if ya_bin_token:
            custom_env['YA_TOKEN'] = ya_bin_token

        custom_env["YA_MERGE_SPLIT_TESTS"] = "yes" if self.options.merge_split_tests else "no"
        custom_env["YA_TEST_DISABLE_FLAKE8_MIGRATIONS"] = "0"

        return custom_env

    def send_cache_hit(self):
        cache_hit_data = load_build_statistics(os.path.join(self.logs_dir, 'build-statistics/cache-hit.json'))
        if cache_hit_data:
            logging.debug('Send cache hit to stream processor, cache_hit data is %s', cache_hit_data)
            if self.stream_client:
                self.stream_client.send_metric('cache_hit', cache_hit_data['cache_hit'], 'NUMBER', 'SUM')

    def send_critical_path(self):
        critical_path_data = load_build_statistics(os.path.join(self.logs_dir, 'build-statistics/critical-path.json'))
        if critical_path_data:
            critical_paths = []
            for critical_path in critical_path_data:
                critical_paths.append({
                    'type': critical_path['type'],
                    'message': critical_path.get('text', ''),
                    'infrastructure': False,
                    'start_ts': critical_path['start_ts'] / 1000,
                    'finish_ts': critical_path['end_ts'] / 1000,
                    'uid': critical_path.get('uid', ''),
                    'hostname': critical_path.get('host', ''),
                    'toolchain': '-'.join(critical_path['tags']) if critical_path.get('tags') else '',
                })

            logging.debug('Send critical paths to stream processor, critical path data is %s', critical_paths)
            if self.stream_client:
                self.stream_client.send_finish()

    def send_build_and_test_slot_time(self):
        slot_time_data = load_build_statistics(os.path.join(self.logs_dir, 'build-statistics/slot_time.json'))
        if slot_time_data:
            slot_time = slot_time_data['slot_time'] / 1000.0  # in seconds
            logging.debug('Send slot time to ci, slot time is %s seconds', slot_time)
            if self.stream_client:
                self.stream_client.send_metric('slot_time', slot_time, 'SECONDS', 'SUM')

    def send_estimated_cpu_time(self, context_path, affected_uids):
        from exts import decompress
        from devtools.distbuild.libs.parse_number.python import parse_number

        with decompress.udopen(context_path) as f:
            context = json.load(f)
        execution_cost = context['graph']['conf'].get('execution_cost', {})
        estimated_execution_cost = execution_cost.get('cpu', 0) or execution_cost.get('cpu_by_minreqs_only', 0)

        logging.debug('Estimated machine hours of the graph: %s', estimated_execution_cost)

        graph = context['graph']['graph']
        affected_uids_set = set(affected_uids)
        affected_estimated_execution_cost = 0
        skiped_affected_uids = 0
        for node in graph:
            if node['uid'] not in affected_uids_set:
                continue
            min_reqs = node.get('min_reqs')
            if not min_reqs:
                skiped_affected_uids += 1
                continue
            node_cpu = min_reqs.get('cpu')
            node_duration = (min_reqs.get('duration') or "0ms")[: -1]

            try:
                affected_estimated_execution_cost += parse_number.parse_human_readable_number(node_cpu) * parse_number.parse_human_readable_number(node_duration)
            except ValueError:
                skiped_affected_uids += 1
                logging.debug('Cannot count executions cost for uid: %s', node['uid'])

        logging.debug('Affected estimated machine hours of the graph: %s', affected_estimated_execution_cost)
        logging.debug('Skip count machine hours of affected uids: %s', skiped_affected_uids)
        logging.debug('Diff between estimated machine hours of the graph and affected uids: ', estimated_execution_cost - affected_estimated_execution_cost)

        if self.stream_client:
            self.stream_client.send_metric('machine_hours', affected_estimated_execution_cost, 'SECONDS', 'SUM')

    def tar_temporary_logs(self, paths, comparing_graph_paths):
        destination = os.path.join(self.logs_dir, 'extra_logs')
        # tar only svn working copies
        if not self.arc_thin_adapter_factory:
            self._tar_temporary_logs([os.path.realpath(paths.source_root)], destination)
            if comparing_graph_paths:
                self._tar_temporary_logs([os.path.realpath(comparing_graph_paths.source_root)], destination)

    def _tar_temporary_logs(self, sources, destination_dir):
        sdk_paths.make_folder(destination_dir)
        for source in sources:
            try:
                with sdk2.helpers.ProcessLog(self.task, logger=logging.getLogger('tar')) as pl:
                    sdk2.helpers.subprocess.run(
                        ['tar', 'cf', os.path.join(destination_dir, os.path.basename(source) + '.tar'), os.path.basename(source)],
                        cwd=os.path.dirname(source),
                        check=True,
                        stdout=pl.stdout,
                        stderr=sdk2.helpers.subprocess.STDOUT
                    )
                logging.debug('Tared %s to %s', source, destination_dir)
            except Exception as e:
                logging.warning('Unable to tar source %s to destination_dir %s because of %s', source, destination_dir, repr(e))

    def _tar_and_clean(self, logs_resource_path):
        if not self.options.save_graph_and_context:
            sdk_paths.remove_path(os.path.join(logs_resource_path, 'context.work.dir', 'graph.json'))
            sdk_paths.remove_path(os.path.join(logs_resource_path, 'context.work.dir', 'context.json'))

        try:
            resource_packager.pack_and_remove([logs_resource_path])
        except Exception as e:
            logging.warning('Unable to tar and clean logs resource directory (%s) because of %s', logs_resource_path, repr(e))

    def start_streaming_server(self, build_state, name_prefix=''):
        if enable_streaming_diractly_to_ci(self.options):
            logging.debug('Skip creating local proxy, send data diractly to ci')
            return

        autocheck_logs_dir = self.logs_dir
        receivers = []
        build_state.receivers = []

        if self.options.use_streaming and self.options.streaming_backends and self.options.streaming_id:
            reports_receiver = stream.StoringReceiver(os.path.join(autocheck_logs_dir, name_prefix + 'reports.json'))
            build_state.receivers.append(reports_receiver)
            raw_reports_receiver = stream.StoringReceiver(os.path.join(autocheck_logs_dir, name_prefix + 'raw_reports.json'))
            receivers.append(raw_reports_receiver)
            build_state.receivers.append(raw_reports_receiver)
            receivers.append(
                stream.CompoundReceiver([
                    reports_receiver,
                    stream.ProxyingReceiver(
                        self.options.streaming_backends,
                        idle_run=self.options.emulate_streaming,
                    )
                ]),
            )
        else:
            self.options.streaming_id = None

        if self.options.report_to_ci:
            try:
                source_id = '{ci_logbroker_source_id}/{partition}'.format(
                    ci_logbroker_source_id=self.options.ci_logbroker_source_id,
                    partition=self.options.partition_index,
                )
                logging.debug('source_id %s', source_id)

                # TODO CI-2673
                check_type = self.options.ci_check_type
                if not check_type:
                    if self.options.circuit_type == 'fast':
                        check_type = "FAST"
                    else:
                        check_type = "FULL"

                ci_client_receiver = stream.CIClientReceiver(
                    logbroker_token=autocheck_tokens.get_logbroker_token(self.task.owner),
                    topic=self.options.ci_logbroker_topic,
                    source_id=source_id,
                    check_id=self.options.ci_check_id,
                    check_type=check_type,
                    number=self.options.ci_iteration_number,
                    partition=self.options.partition_index,
                    task_id_string=self.options.ci_task_id,
                    skip_results=self.options.emulate_streaming or not self.is_autocheck_20,
                    sandbox_task_id=self.task.id,
                )
                receivers.append(ci_client_receiver)
                build_state.receivers.append(ci_client_receiver)
            except Exception:
                logging.exception('Error in CI Client')

        if receivers:
            main_receiver = stream.CompoundReceiver(receivers)
            build_state.report_server = stream.StreamReportServer(main_receiver)
            build_state.report_server_url = build_state.report_server.url
            logging.debug('Streaming server started at %s', build_state.report_server_url)

    def trace_stage(self, stage):
        try:
            if self.stream_client:
                self.stream_client.trace_stage('distbuild/{}'.format(stage))
        except Exception:
            logging.exception('Finally failed to trace stage')

    def load_profile(self, start_time, profile_prefix, profile_path):
        if os.path.exists(profile_path):
            profile = json.load(open(profile_path, 'r'))
            if 'step' in profile:
                for k, v in six.iteritems(profile['step']):
                    self.update_time_metrics(profile_prefix + k, max(v, 0) + start_time - self.commit_time, sync=False)
            if 'value' in profile:
                for k, v in six.iteritems(profile['value']):
                    self.update_time_metrics(profile_prefix + k, v, sync=False)

        times = self.context.time_metrics
        self.context.configure_work_time = times.get('configure_finished', 0) - times.get('configure_started', 0)
        self.context.cmake_work_time = times.get('configure_finished', 0) - times.get('configure_started', 0)
        self.context.tests_work_time = times.get('testing_finished', 0) - times.get('testing_started', 0)
        self.context.make_work_time = times.get('build_finished', 0) - times.get('build_started', 0)

    def modify_stages(self, start_time, stage_prefix, stages_path):
        stages = {}
        if os.path.exists(stages_path):
            with open(stages_path) as f:
                stages = json.load(f)
            stages = {stage_prefix + name: max(t, start_time) for name, t in six.iteritems(stages)}
            with open(stages_path, 'w') as f:
                json.dump(stages, f, indent=4)
            logging.debug('Modified timestamp for stages with timestamp before start_time to start_time, stages path %s', stages_path)
        return stages_path, stages

    def load_graph_stages_and_profile(self, comparing_graph_build_state, build_state):
        try:
            if all([
                comparing_graph_build_state is not None,
                comparing_graph_build_state.build_graph_last_stage is not None,
                build_state.build_graph_last_stage is None or comparing_graph_build_state.build_graph_last_stage > build_state.build_graph_last_stage
            ]):
                last_build_state = comparing_graph_build_state
            else:
                last_build_state = build_state

            last_build_state.build_graph_stages_path, _ = self.modify_stages(last_build_state.start_time, 'graph-', last_build_state.build_graph_stages_path)
            self.load_profile(last_build_state.start_time, 'graph-', last_build_state.build_graph_profile_path)
            self.load_stages(last_build_state.build_graph_stages_path)
        except Exception as e:
            logging.debug('Failed to update stages, error: %s', repr(e))

    def run_build(self, logs_resource_path, options, build_state, paths, repo, build_mode, log_prefix='build'):
        logging.debug('running build in %s mode', build_mode)
        test_output_dir = os.path.join(paths.room_path, 'build-release')
        test_output_link = os.path.join(logs_resource_path, 'build-release')
        sdk_paths.make_folder(test_output_dir)
        os.symlink(test_output_dir, test_output_link)

        compiler_messages_dir = os.path.join(logs_resource_path, 'compiler-messages')
        compiler_messages_link = os.path.join(logs_resource_path, 'compiler-messages.tar', 'compiler-messages')
        sdk_paths.make_folder(os.path.dirname(compiler_messages_link))
        sdk_paths.make_folder(compiler_messages_dir)
        os.symlink(compiler_messages_dir, compiler_messages_link)

        error_file = str(self.log_path('error.txt'))

        builder = ymake.YaBuilder(
            paths.source_root, os.path.join(paths.room_path, 'build_root'),
            logs_resource_path
        )

        use_custom_context = (build_mode == BuildMode.ONLY_BUILD) and options.use_custom_context

        cpu_count = int(sdk_util.system_info()['ncpu'])
        options.make_context_on_distbuild = options.make_context_on_distbuild and build_mode not in [BuildMode.ONLY_BUILD]

        ya_options = options.make_ya_options(
            builder.ya_cmd,
            builder.source_root,
            builder.build_root,
            builder.output_dir,
            test_output_dir,
            build_threads=cpu_count,
            coordinators_filter=options.coordinators_filter,
            custom_context=build_state.context_path if use_custom_context else None,
            distbs_pool=options.distbs_pool,
            error_file=error_file,
            patch_spec=options.arc_patch,
            revision=repo.revision,
            streaming_url=build_state.report_server_url,
            svn_url=repo.reference.svn_url,
            arc_url_as_working_copy_in_distbuild=repo.reference.arc_url,
            task_id=self.task.id,
            enable_streaming_diractly_to_ci=enable_streaming_diractly_to_ci(self.options),
            ci_logbroker_partition_group=self.ci_logbroker_partition_group,
            ci_logbroker_token=autocheck_tokens.get_logbroker_token(self.task.owner) if enable_streaming_diractly_to_ci(self.options) else None,
        )

        build_process = builder.build(
            ya_options,
            wait_process=True,
            strace=str(self.path('strace.log')) if options.use_strace else False,
            # log_prefix=log_prefix,
            stdout=open(os.path.join(logs_resource_path, '{}.out.txt'.format(log_prefix)), 'w'),
            stderr=open(os.path.join(logs_resource_path, '{}.err.txt'.format(log_prefix)), 'w'),
        )

        def on_build_finished(build_returncode):
            self.stage_started('build-task-finalization')
            builder.on_build_process_finished(build_returncode)

            if build_returncode != 0:
                error_message = None
                if os.path.isfile(error_file):
                    with open(error_file) as ef:
                        error_message = ef.read().strip()

                sdk_paths.remove_path(test_output_link)
                setattr(self.context, 'ya_error_message', str(error_message))
                raise AutocheckBuildError('Build return code is {}, see build.out.txt for details, message is: {}'.format(build_returncode, error_message))

            profile_path = os.path.join(logs_resource_path, 'profile.json')
            original_stages_path = os.path.join(logs_resource_path, 'stages.json')
            stages_path, stages = self.modify_stages(build_state.start_time, '', original_stages_path)

            self.load_profile(build_state.start_time, '', profile_path)
            self.load_stages(stages_path)

            self.stage('last-ya-make-stage-finished-for-{}'.format(log_prefix), max(stages.values()) if stages else None)

        on_build_finished(build_process.returncode)
        return builder
