import Queue
import collections
import logging
import multiprocessing
import time
import traceback

from sandbox import sdk2
import sandbox.common.types.resource as ctr

from sandbox.projects.autocheck.lib.core import resources
from sandbox.projects.autocheck.lib.deprecated import process_results

import subtask


def clear_resource(task, partition, partition_path):
    logging.debug('Clear resource and work dir for partition %d', partition)
    last_autocheck_resource = resources.AutocheckLogs.find(task=task, attrs={'partition': partition}).first()
    if last_autocheck_resource and last_autocheck_resource.state == ctr.State.NOT_READY:
        sdk2.ResourceData(last_autocheck_resource).broken()
    last_meta_graph_resource = resources.YaMakeMetaGraphResult.find(task=task, attrs={'partition_index': partition}).first()
    if last_meta_graph_resource and last_meta_graph_resource.state == ctr.State.NOT_READY:
        sdk2.ResourceData(last_meta_graph_resource).broken()
    sdk2.paths.remove_path(partition_path)


def terminate_all_subtasks(task, subtasks, lock):
    with lock:
        for partition, proc in subtasks.items():
            info = task.Context.subtasks_info[str(partition)]
            if info['finished']:
                continue
            proc.terminate()
            clear_resource(task, partition, proc.path())


def run_subtasks(task):
    if not task.Context.subtasks_info:
        for partition in range(task.Context.projects_partitions_count):
            task.Context.subtasks_info[str(partition)] = {
                'finished': False,
            }
        task.Context.save()

    subtasks = {}
    restarts = {}
    errors = collections.defaultdict(set)
    partition_times = collections.defaultdict(dict)  # {partition: {'start':[], 'finish:[]}}
    queue = multiprocessing.Queue()
    lock = multiprocessing.Lock()

    logging.info('Start subtasks in processes')
    for partition in range(task.Context.projects_partitions_count):
        if task.Context.subtasks_info[str(partition)]['finished']:
            continue
        parameters = task.create_task_parameters(partition)
        proc = subtask.AutocheckBuildYa2Subtask(partition, task, parameters, task.Context.__GSID, lock, queue)
        proc.start()
        subtasks[partition] = proc
        restarts[partition] = 0
        partition_times[partition]['start'] = [time.time()]
        partition_times[partition]['finish'] = []
        logging.info('Start partition %d as process %d', partition, proc.pid)

    while True:
        if all(info['finished'] for _, info in task.Context.subtasks_info.items()):
            break

        try:
            sub_status = queue.get(timeout=300)
        except Queue.Empty:
            with lock:
                for partition, proc in subtasks.items():
                    info = task.Context.subtasks_info[str(partition)]
                    if info['finished']:
                        continue
                    if not proc.is_alive():
                        logging.debug('Process %d of Partition %d ended unexpectedly', proc.pid, partition)
                        clear_resource(task, partition, proc.path())
                        err = 'Partition {} ended unexpectedly'.format(partition)
                        queue.put(subtask.SubtaskStatus.prepare(partition, subtask.SubtaskStatus.FAILED, proc.context, err))
            continue
        partition_times[sub_status.partition]['finish'].append(time.time())
        if sub_status.code == subtask.SubtaskStatus.SUCCESS:
            logging.info('Partition %d was ended successfully in process %d', sub_status.partition, subtasks[sub_status.partition].pid)
            with lock:
                task.Context.subtasks_info[str(sub_status.partition)]['finished'] = True
                task.Context.save()
            try:
                process_results.process_results(task, partition_times[sub_status.partition], sub_status)
            except Exception:
                logging.debug('Fail to process results for partition %d:\n%s', sub_status.partition, traceback.format_exc())
        elif sub_status.code == subtask.SubtaskStatus.PESSIMIZED:
            logging.info('Partition %d exceeds limit by nodes, stop task', sub_status.partition)
            with lock:
                task.Context.should_pessimize_by_nodes = True
                task.Context.save()
            terminate_all_subtasks(task, subtasks, lock)
            raise sdk2.WaitTime(1)
        else:
            err_lines = sub_status.err_message.strip().split('\n')
            if len(err_lines) > 10:
                err_message = '\n'.join(err_lines[:4] + ['...'] + err_lines[-5:])
            else:
                err_message = sub_status.err_message

            if sub_status.retry_partition is False:
                logging.info('Partition %d is stopped', sub_status.partition)
                with lock:
                    task.Context.subtasks_info[str(sub_status.partition)]['finished'] = True
                    task.Context.save()
                    task.set_info('Partition {} is stopped, reason:\n{}'.format(sub_status.partition, err_message))
                continue

            if sub_status.err_message not in errors[sub_status.partition]:
                errors[sub_status.partition].add(sub_status.err_message)
                with lock:
                    task.set_info('Partition {} failed with exception:\n{}'.format(sub_status.partition, err_message))
            if restarts[sub_status.partition] > task.Context.max_subtask_count:
                logging.info('Partition %d was failed in process %d', sub_status.partition, subtasks[sub_status.partition].pid)
                with lock:
                    for _, proc in subtasks.items():
                        proc.terminate()
                raise Exception('No child tasks to wait for partition {}'.format(sub_status.partition))

            logging.info('Partition %d is restarting', sub_status.partition)
            proc = subtasks[sub_status.partition]
            if proc.is_alive():
                proc.join()
            restarts[sub_status.partition] += 1
            partition_times[sub_status.partition]['start'].append(time.time())
            parameters = task.create_task_parameters(sub_status.partition)
            new_proc = subtask.AutocheckBuildYa2Subtask(sub_status.partition, task, parameters, task.Context.__GSID, lock, queue)
            subtasks[sub_status.partition] = new_proc
            new_proc.start()
            logging.info('Patition %d was restarted in process %d', sub_status.partition, subtasks[sub_status.partition].pid)

    for partition, proc in subtasks.items():
        proc.join()
        logging.info('Partition %d exited with code %d', partition, proc.exitcode)
