import datetime
import logging
import time

from sandbox import sdk2
from sandbox.projects.autocheck.AutocheckBuildParent2 import AutocheckBuildParent2


DATETIME_REGEXP = r'\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}'
DATETIME_FMT = 'YYYY-mm-ddTHH:MM:SS'


class AutocheckReplay(sdk2.Task):

    class Requirements(sdk2.Requirements):
        cores = 1
        disk_space = 1024
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Autocheck tasks querying parameters') as query_block:
            taskq_start_time = sdk2.parameters.StrictString('Start datetime (%s)' % DATETIME_FMT,
                                                            regexp=DATETIME_REGEXP,
                                                            required=True,
                                                            default=datetime.datetime.today().strftime('%Y-%m-%dT00:00:00'))
            taskq_end_time = sdk2.parameters.StrictString('End datetime (%s)' % DATETIME_FMT,
                                                          regexp=DATETIME_REGEXP,
                                                          required=True,
                                                          default=datetime.datetime.today().strftime('%Y-%m-%dT00:00:00'))
            taskq_limit = sdk2.parameters.Integer('Tasks limit', default=100000, required=True)
            taskq_postcommit = sdk2.parameters.Bool('Postcommit', default=False, required=True)
            taskq_tag = sdk2.parameters.String('Additional tag to add to query', default=None, required=False)
        with sdk2.parameters.Group('Autocheck tasks replaying parameters') as replay_block:
            taskr_time_limit = sdk2.parameters.Integer('Time limit (sec)', default=3600)
            taskr_is_rate_fixed = sdk2.parameters.Bool('Fixed rate', default=False, required=True)
            with taskr_is_rate_fixed.value[False]:
                taskr_rate_coef = sdk2.parameters.Float('Original rate multiplier', default=1.0, required=True)
            with taskr_is_rate_fixed.value[True]:
                taskr_rate = sdk2.parameters.Float('Rate (1/sec)', default=0.1, required=True)
            taskr_skip_tests = sdk2.parameters.Bool('Do not run tests', default=False, required=True)

    class Context(sdk2.Context):
        pass

    def on_execute(self):
        super(AutocheckReplay, self).on_execute()

        logging.info('Querying for tasks...')
        searchq = {
            'type': ['AUTOCHECK_BUILD_PARENT_2'],
            'created': '{}..{}'.format(self.Parameters.taskq_start_time, self.Parameters.taskq_end_time),
            'all_tags': True,
            'hidden': True,
        }
        if self.Parameters.taskq_postcommit:
            searchq.update({
                'tags': ['TESTENV-AUTOCHECK-JOB', 'TESTENV-COMMIT-CHECK'],
            })
        else:
            searchq.update({
                'tags': ['TESTENV-AUTOCHECK-JOB', 'TESTENV-PRECOMMIT-CHECK'],
            })

        if self.Parameters.taskq_tag:
            searchq['tags'].append(self.Parameters.taskq_tag)

        tasks = sdk2.Task.find(**searchq).order(sdk2.Task.created).limit(self.Parameters.taskq_limit)

        logging.info('Iterating over tasks...')
        t0 = None
        orig_t0 = None
        for i, task in enumerate(tasks):
            new_task = AutocheckBuildParent2(None, **dict(task.Parameters))
            new_task.Parameters.description = 'Created from #{} by #{}'.format(task.id, self.id)
            new_task.Parameters.owner = self.owner
            new_task.Parameters.priority = task.Parameters.priority
            new_task.Parameters.autocheck_distbs_testing_cluster = True
            new_task.Parameters.use_streaming = False
            new_task.Parameters.report_to_ci = False
            new_task.Parameters.send_logs_to_logbroker = False
            new_task.Parameters.use_dist_diff = True
            new_task.Parameters.semaphores = '[]'
            new_task.Parameters.coordinators_filter_map = ''
            new_task.Parameters.pool_name = ''
            new_task.Parameters.save_meta_graphs = False
            new_task.Parameters.ya_bin_token = 'YA_TOKEN:robot-distbuild-test'
            if self.Parameters.taskr_skip_tests:
                new_task.Parameters.run_tests = False
                new_task.Parameters.test_sizes = []
                new_task.Parameters.test_type_filters = []

            if i:
                assert t0
                if self.Parameters.taskr_is_rate_fixed:
                    ti = t0 + datetime.timedelta(seconds=(i / self.Parameters.taskr_rate))
                else:
                    assert orig_t0
                    orig_ti = task.created
                    ti = t0 + datetime.timedelta(seconds=((orig_ti - orig_t0).total_seconds() / self.Parameters.taskr_rate_coef))
                now = datetime.datetime.now()
                if ti > now:
                    delay = (ti - now).total_seconds()
                    logging.info('Sleeping for %f secs', delay)
                    time.sleep(delay)
                else:
                    logging.warn('Unable to run task %d in time (lag = %s)', i, str(now - ti))
            else:
                t0 = datetime.datetime.now()
                orig_t0 = task.created

            new_task.save().enqueue()
            logging.info('Enqueued task %d (#%d) as #%d', i, task.id, new_task.id)

        logging.info('All done')
