import re
import six


class Error(object):
    ERROR_RE = None

    def __init__(self):
        self.summary = 'The check will be incomplete due to autocheck internal errors'
        self._message = ''

    @property
    def message(self):
        return self._message

    @message.setter
    def message(self, msg):
        if len(msg) > 2000:
            self._message = msg[: 2001]
        else:
            self._message = msg

    def none_retriable(self, message):
        if self.ERROR_RE:
            return self._search(self.ERROR_RE, message)
        return False

    def _search(self, regexp, text):
        text = six.ensure_str(text)
        return bool(regexp.search(text))


class DistbuildRepositoryError(Error):
    ERROR_RE = re.compile('Repository acquisition failed')

    def __init__(self):
        super(DistbuildRepositoryError, self).__init__()
        self.message = 'Failed to prepare repository on distbuild'


class DistbuildResourceError(Error):
    ERROR_RE = re.compile('Resource acquisition failed')

    def __init__(self):
        super(DistbuildResourceError, self).__init__()
        self.message = 'Failed to prepare resources on distbuild'


class DistbuildRemoteCommandError(Error):
    ERROR_RE = re.compile('RemoteCommandError:')

    def __init__(self):
        super(DistbuildRemoteCommandError, self).__init__()
        self.message = 'Failed to execute command on distbuild'


class DistbuildAuthError(Error):
    ERROR_RE = re.compile('Build is failed, error: Build rejected: Unknown tree')

    def __init__(self):
        super(DistbuildAuthError, self).__init__()
        self.message = 'Failed to authenticate user on distbuild'


class FastCircuitError(Error):
    ERROR_RE = re.compile('Fail fast circuit')

    def __init__(self):
        super(FastCircuitError, self).__init__()
        self.message = 'Fast circuit failed'


class SandboxResourceError(Error):
    ERROR_RE = re.compile('Invalid sandbox resource id')

    def __init__(self):
        super(SandboxResourceError, self).__init__()
        self.message = 'Invalid sandbox resource id'


class YaArgsBindingError(Error):
    ERROR_RE = re.compile('ArgsBindingException: Do not know what to do with')

    def __init__(self):
        super(YaArgsBindingError, self).__init__()
        self.message = 'Unknown arguments in the `ya make` command'


class YaConfError(Error):
    ERROR_RE = re.compile('JSONDecodeError:')

    def __init__(self):
        super(YaConfError, self).__init__()
        self.message = 'ya failed to load ya.conf.json'


class YcmergeError(Error):
    ERROR_RE = re.compile('ycmerge: error:')

    def __init__(self):
        super(YcmergeError, self).__init__()
        self.message = 'ycmerge failed'


class YmakeCrashedError(Error):
    ERROR_RE = re.compile('YMake crashed')

    def __init__(self):
        super(YmakeCrashedError, self).__init__()
        self.message = 'ymake crashed'


class YmakeFailedError(Error):
    ERROR_RE = re.compile('YMake failed with exit code 1')

    def __init__(self):
        super(YmakeFailedError, self).__init__()
        self.message = 'ymake failed to build graph'


class ZipatchApplyError(Error):
    ERROR_RE = re.compile('ZipatchMalformedError:')

    def __init__(self):
        super(ZipatchApplyError, self).__init__()
        self.message = 'Failed to apply a patch to the repository'


Errors = [
    DistbuildAuthError,
    DistbuildRemoteCommandError,
    DistbuildRepositoryError,
    DistbuildResourceError,
    FastCircuitError,
    SandboxResourceError,
    YaArgsBindingError,
    YaConfError,
    YcmergeError,
    YmakeCrashedError,
    YmakeFailedError,
    ZipatchApplyError,
]


def is_none_retriable(message):
    message = six.ensure_str(message)
    for error_class in Errors:
        er = error_class()
        if er.none_retriable(message):
            return er

    return None
