import logging
import os
import tarfile
import tempfile

from sandbox import sdk2


class AdapterError(Exception):
    pass


def is_arc_url(arcadia_url):
    return arcadia_url.startswith('arcadia-arc:/')


def get_arc_binary_from_resource(task):
    arc_client_resource = sdk2.Resource.find(
        type='ARC_CLIENT',
        attrs={'platform': 'linux', 'released': 'stable'}
    ).first()

    logging.info('arc client resource: %s', arc_client_resource)
    if not arc_client_resource:
        return None

    data = sdk2.ResourceData(arc_client_resource)
    temp_dir = tempfile.mkdtemp(prefix='arc_client')
    with tarfile.open(str(data.path)) as tar:
        tar.extractall(temp_dir)

    arc_client_path = os.path.join(temp_dir, 'arc')
    if not os.path.exists(arc_client_path):
        logging.info('Cannot find arc client at path %s', arc_client_path)
        return None

    logging.info('Use arc client from resource: %s, at path: %s', arc_client_resource, arc_client_path)
    return arc_client_path


def get_arc_adapter_factory(path, arc_bin, token):
    # url in format:
    #   svn:      arcadia:/arc/...@revision
    #   arc:      arcadia-arc:/#{revision | hash}
    # examples:
    #   arcadia:/arc/trunk/arcadia@8478937
    #   arcadia:/arc/branches/yabs/server/r551/arcadia@8468110
    #   arcadia-arc:/#trunk
    #   arcadia-arc:/#releases/experimental/release_machine/release_machine_test/stable-12582
    #   arcadia-arc:/#621ad8839dbc2897c3ff0c8fc4329d30b9968bf3
    import yalibrary.ggaas.bare_repository as repo
    import yalibrary.ggaas.interface.repo as i

    class ArcRepoAdapterFactory(object):
        def __init__(self, path, arc_bin, token):
            self.repository_config = repo.RepositoryConfig(
                work_dir=path,
                token=token,
                arc_binary=arc_bin,
            )

        def make_arc_thin_adapter_factory(self):
            return ArcThinAdapterFactory(self.repository_config)

        def unmount(self):
            self.repository_config.cleanup()

        def __enter__(self):
            return self

        def __exit__(self, exc_type, exc_val, exc_tb):
            self.unmount()

    class ArcThinAdapterFactory(object):
        def __init__(self, repository_config):
            self.repository_config = repository_config
            self._thin_repos = []  # type: list[repo.ArcThinArcadiaRepository]

        def make_arc_adapter(self, path, url, zipatch):
            thin_repo = repo.ArcThinArcadiaRepository(
                working_copy=i.WorkingCopy(url, i.PatchURL(zipatch) if zipatch else None),
                repository_config=self.repository_config,
                work_dir=path,
            )
            thin_repo.setup()

            self._thin_repos.append(thin_repo)
            return thin_repo

        def __enter__(self):
            return self

        def __exit__(self, exc_type, exc_val, exc_tb):
            for thin_repo in self._thin_repos:
                thin_repo.finalize()

    return ArcRepoAdapterFactory(path, arc_bin, token)
