import collections
import copy
import hashlib
import json
import logging
import os


Target = collections.namedtuple('Target', ['path', 'uid'])


class Build(object):
    def __init__(self, path, toolchain, targets):
        self.path = path
        self.toolchain = toolchain
        self.targets = targets

    @staticmethod
    def from_json(params):
        path = params['path']
        toolchain = params['toolchain']
        targets = [Target(path=p, uid=None) for p in params['targets']]
        return Build(path, toolchain, targets)

    def __str__(self):
        return '{}:{}:{}'.format(self.path, self.toolchain, ','.join([str(target) for target in self.targets]))


class NativeBuilds(object):
    def __init__(self, params):  # expect str([{"path": "path", "toolchain": "toolchain", targets: [paths]}])
        self.builds = [Build.from_json(parameters) for parameters in json.loads(params)]
        self.bin_result_path = None

    def combine_targets(self):
        paths = [target.path for build in self.builds for target in build.targets]
        logging.debug('Native builds combine targets: %s', paths)
        return paths

    def set_bin_result_path(self, bin_result_path):
        self.bin_result_path = bin_result_path

    def filter_builds(self):
        if not self.bin_result_path:
            logging.waring('bin_result_path is not set, return empty list of filtered builds')
            return []

        found_targets = self._load_found_targets()
        logging.debug('Discovered native builds targets: %s', found_targets)

        filtered_builds = []
        for build in self.builds:
            filtered_targets = []
            for target in build.targets:
                for found_target in found_targets:
                    if _is_subpath(found_target.path, target.path):
                        filtered_targets.append(found_target)
            if filtered_targets:
                new_build = copy.deepcopy(build)
                new_build.targets = filtered_targets
                filtered_builds.append(new_build)
        return filtered_builds

    def run_native_builds(self, streaming_client, revision, precommit):
        filtered_builds = self.filter_builds()
        payload = []
        for build in filtered_builds:
            payload.append({
                'status': 'DISCOVERED',
                'owners': {
                    'logins': [],
                    'groups': [],
                },
                'toolchain': build.toolchain,
                'uid': self._create_test_uid(build),  # hash from build with uids
                'links': {},
                'tags': [
                    'ya:force_sandbox',
                    'ya:fat'
                ],
                'error_type': None,
                'requirements': {
                    'sb_vault': 'ARC_TOKEN_PATH=file:YATOOL:ARC_TOKEN',
                    'network': 'full'
                },
                'name': 'pytest',
                'subtest_name': 'native:{}'.format(','.join([target.path for target in build.targets])),
                'rich-snippet': '',
                'suite_id': None,
                'result': {},
                'suite': True,
                'duration': None,
                'path': build.path,
                'type': 'test',
                'id': _hash(':'.join([build.path, build.toolchain] + [target.path for target in build.targets] + ['precommit'] if precommit else [])),
                'size': 'large',
            })
        logging.debug('Send payload with native builds: %s', payload)
        streaming_client.send_chunk(payload)

    def _load_found_targets(self):
        '''
            bin_result_path can contain several elements with the same path and different uids
            we need to determine that a commit affects target, so we get only first such path
            with uid and skip others
        '''
        loaded_paths = set()
        found_targets = []
        with open(self.bin_result_path) as f:
            for path_uid_string in sorted(json.load(f)):
                path, uid = path_uid_string.split(':', 1)
                if path not in loaded_paths:
                    found_targets.append(Target(path=path, uid=uid))
                    loaded_paths.add(path)
        return found_targets

    def _create_test_uid(self, build):
        target_string = str(build)
        logging.debug('Create uid for native build large test from %s', target_string)
        uid = _hash(target_string)
        return uid


def _is_subpath(path, root):
    def skip_last_sep(p):
        if p.endswith(os.path.sep):
            return p[: -1]
        return p

    path = skip_last_sep(path)
    root = skip_last_sep(root)

    path_chunks = path.split(os.path.sep)
    root_chunks = root.split(os.path.sep)
    i = 0
    while i < len(root_chunks) and root_chunks[i] == path_chunks[i]:
        i += 1

    return i == len(root_chunks)


def _hash(string):
    m = hashlib.md5()
    m.update(string)
    return m.hexdigest()
