import contextlib
import logging
import multiprocessing as mp
import six
import typing as tp  # noqa

from sandbox import sdk2


def default_format_entry(threadname, fname, line, fun, fmt="%(fun)s"):
    return fmt % locals()


def custom_format_entry(threadname, fname, line, fun, fmt="%(fname)s:%(line)s %(fun)s"):
    return fmt % locals()


class FlamegraphProfiler(object):
    def __init__(self, work_dir):
        self._work_dir = work_dir
        sdk2.paths.make_folder(self._work_dir, delete_content=True)
        self._perf_logs = self._work_dir / 'perfs'
        sdk2.paths.make_folder(self._perf_logs)
        self._report_path = None
        self._lock = mp.Lock()
        self._logger = logging.getLogger('FlamegraphProfiler')

    @contextlib.contextmanager
    def start_profile_thread(self, title=None, format_entry=custom_format_entry):
        from logos.libs.profiling.flamegraph import flamegraph

        filepath = self._get_perf_log_name(title)
        self._logger.info('Start profiling to %s', str(filepath))
        fd = open(str(filepath), 'w')
        ft = flamegraph.start_profile_thread(fd=fd, format_entry=format_entry)
        try:
            yield
        finally:
            self._logger.info('Stop profiling to %s', str(filepath))
            ft.stop()
            fd.close()

    def prepare_report(self, task):
        try:
            return self._prepare_report(task)
        except:
            self._logger.exception('Fail to prepare flamegraph report')
            return None, None

    def _prepare_report(self, task):
        with self._lock:
            content = []
            for filename in sorted(self._perf_logs.iterdir()):
                svg_path = self._make_svg(self._perf_logs / filename)
                if svg_path is None:
                    self._logger.error('Fail to prepare flamegraph from %s', self._perf_logs / filename)
                    continue

                url = '{proxy}/{path}'.format(proxy=task.log_resource.http_proxy, path=svg_path.relative_to(task.log_path()))
                content += [
                    '<div>',
                    '<object type="image/svg+xml" data="{url}">'.format(url=url),
                    '<img src="{name}" />'.format(name=svg_path.with_suffix('.png').name),
                    '</object>',
                    '</div>',
                ]
            report_path = self._work_dir / 'report.html'
            with open(str(report_path), 'w') as f:
                f.write('\n'.join(content))
            report_link = '{proxy}/{path}'.format(proxy=task.log_resource.http_proxy, path=report_path.relative_to(task.log_path()))
            return report_path, report_link

    def _make_svg(self, perf_log_path):
        flame_pict = self._work_dir / perf_log_path.with_suffix('.svg').name
        self._logger.info('Prepare flamegraph to %s', str(flame_pict))
        flamegraph_pl = self._get_flamegraph_pl()
        if not flamegraph_pl:
            self._logger.error('Fail prepare flamegraph: cannot find flamegraph.pl')
            return None
        with open(str(flame_pict), 'w') as f:
            p = sdk2.helpers.subprocess.run([str(flamegraph_pl), '--title', 'AutocheckTask: {}'.format(flame_pict.stem), str(perf_log_path)], stdout=f)
            if p.returncode != 0:
                return None
        return flame_pict

    def _get_perf_log_name(self, title=None):
        with self._lock:
            if title is not None:
                return self._perf_logs / '{title}.log'.format(title=title)
            count = len(list(self._perf_logs.glob('perf_*'))) + 1
            return self._perf_logs / 'perf_{count}.log'.format(count=count)

    def _get_flamegraph_pl(self):
        import library.python.resource as rs

        flamegraph_pl = self._work_dir / 'flamegraph.pl'
        if flamegraph_pl.exists():
            return flamegraph_pl

        content = rs.find('/flamegraph.pl')
        if not content:
            return None

        with open(str(flamegraph_pl), 'w') as f:
            f.write(six.ensure_str(content))
        flamegraph_pl.chmod(0o755)
        return flamegraph_pl
