import json
import logging
import random
import six
import socket
import threading
import time
import traceback

from six.moves import http_client
from six.moves import urllib
from six.moves.BaseHTTPServer import BaseHTTPRequestHandler, HTTPServer

from sandbox.common import patterns

_STREAMING_REQUEST_TIMEOUT = 180


def _get_host_name():
    s = socket.socket(socket.AF_INET6)
    s.connect(('sandbox.yandex-team.ru', 80))
    addr = s.getsockname()[0]
    logging.debug('Hostname is %s', addr)
    return str(addr)


class HTTPError(urllib.error.HTTPError):
    def __init__(self, error):
        self.code = error.code
        self.text = error.read()

    def __str__(self):
        return 'HTTPError {}'.format(self.code)


class HTTPServerV6(HTTPServer):
    address_family = socket.AF_INET6 if socket.has_ipv6 else HTTPServer.address_family


def make_handler(callback):
    class ReportHandler(BaseHTTPRequestHandler):
        _logger = logging.getLogger('ReportHandler')

        def log_message(self, *args, **kwargs):
            self._logger.debug(*args, **kwargs)

        def do_POST(self):
            self._handle('POST')

        def do_GET(self):
            self._handle('GET')

        def do_PATCH(self):
            self._handle('PATCH')

        def _handle(self, method):
            try:
                if six.PY3:
                    length = int(self.headers.get('content-length'))
                else:
                    length = int(self.headers.getheader('content-length'))
                self._logger.debug('Incoming request to %s with method %s from %s, length is %s', self.path, method, self.address_string(), length)
                message = self.rfile.read(length)
                data = json.loads(message)
                self._response(http_client.OK, callback(method, self.path, data))
            except HTTPError as e:
                self._logger.debug('Request to %s with method %s from %s failed with HttpError, code %s, error is %s', self.path, method, self.address_string(), e.code, traceback.format_exc())
                self._response(e.code, e.text)
            except Exception as e:
                self._logger.debug('Request to %s with method %s from %s failed, error is %s', self.path, method, self.address_string(), traceback.format_exc())
                self._response(http_client.INTERNAL_SERVER_ERROR, repr(e))

        def _response(self, http_code, message):
            message = six.ensure_binary(message or '')
            self._logger.debug('Response to request to %s from %s, with code %s and message %s', self.path, self.address_string(), http_code, message)
            self.send_response(http_code)
            self.send_header('Content-type', 'text/plain')
            self.send_header('Content-Length', len(message))
            self.end_headers()
            self.wfile.write(message)

    return ReportHandler


class StreamReportServer(object):
    def __init__(self, callback, port=None):
        self._logger = logging.getLogger('StreamReportServer')
        self._port = port or random.randint(40000, 50000)
        self._event = threading.Event()
        self._callback = callback
        self._server_thread = None
        self._running = True
        self._server = None
        self._start()

    def _run_server(self):
        for retry in range(1000):
            try:
                server = HTTPServerV6(('', self._port), make_handler(self._callback))
                self._logger.info('Server is started on port %s', self._port)
                self._event.set()
                return server
            except Exception as e:
                self._logger.warn('Unable to start server with port %s, error is %s', self._port, repr(e))
                self._port += 1
        return None

    def _start(self):
        def handle_request():
            try:
                self._server = self._run_server()
                self._server.timeout = 1
                while self._running:
                    self._server.handle_request()
                self._logger.info('Server is finished')
            except Exception:
                self._logger.warn('Server is failed, %s', traceback.format_exc())

        self._server_thread = threading.Thread(target=handle_request)
        self._server_thread.setDaemon(True)
        self._server_thread.start()

    def port(self):
        self._event.wait()
        return self._port

    @patterns.singleton_property
    def url(self):
        return 'http://{}:{}'.format(_get_host_name(), self.port())

    def stop(self):
        self._running = False
        self._server_thread.join()
        self._logger.info('Server is stopped')


class CompoundReceiver(object):
    def __init__(self, callbacks):
        self._logger = logging.getLogger('CompoundReceiver')
        self._callbacks = callbacks

    def __call__(self, method, path, data):
        error = None
        result = None
        for cb in self._callbacks:
            try:
                cb_res = cb(method, path, data)
                if result is None:
                    result = cb_res
            except Exception as e:
                self._logger.warn('Callback is failed, %s', traceback.format_exc())
                error = e

        if error is not None:
            raise error
        return result


class StoringReceiver(object):
    def __init__(self, path):
        self._logger = logging.getLogger('StoringReceiver')
        self._mutex = threading.RLock()
        self._file = open(path, 'w')

    def __call__(self, method, path, data):
        with self._mutex:
            json.dump(data, self._file, separators=(',', ':'))
            self._file.write('\n')
            self._file.flush()
            self._logger.debug('Store %s results with method %s, path %s', len(data.get('results', [])), method, path)

    def close(self):
        self._file.close()


class ProxyingReceiver(object):
    def __init__(self, backends, idle_run=False):
        self._logger = logging.getLogger('ProxyingReceiver')
        self._backends = backends
        self._counter = 0
        self._errors_counter = 0
        self._idle_run = idle_run

    def _request(self, method, url, path, message):
        full_url = url + path
        self._logger.debug('Request to %s with method %s and %s bytes (idle=%s)', full_url, method, len(message), self._idle_run)
        if self._idle_run:
            return

        request = urllib.request.Request(full_url, data=message, headers={'Content-Type': 'application/json'})
        request.get_method = lambda: method
        response = urllib.request.urlopen(request, timeout=_STREAMING_REQUEST_TIMEOUT)
        return response.read()

    def _getUrl(self):
        if self._counter == len(self._backends):
            self._counter = 0
            self._errors_counter = 0

        url = self._backends[self._counter % len(self._backends)]
        self._counter += 1
        return url

    def __call__(self, method, path, data):
        start_time = time.time()
        url = self._getUrl()
        try:
            message = json.dumps(data, separators=(',', ':'))
            self._logger.debug('Proxying message with method %s from %s of length %s to url %s', method, path, len(message), url)
            response = self._request(method, url, path, message)
            self._logger.debug('Response for proxying message with method %s from %s is %s, elapsed %s', method, path, response, time.time() - start_time)
            return response
        except Exception as e:
            self._logger.debug('Unable to proxy message with method %s from %s, elapsed %s', method, path, time.time() - start_time)
            if isinstance(e, urllib.error.HTTPError):
                error = HTTPError(e)
                self._logger.debug('Error is %s, code is %s', error.text, error.code)
            else:
                self._logger.exception('Error: ')
            self._errors_counter += 1
            raise


class CIClientReceiver(object):
    def __init__(self, logbroker_token, topic, source_id, check_id, check_type,
                 number, partition, task_id_string, skip_results, sandbox_task_id):
        self.skip_results = skip_results
        try:
            from testenv.autocheck.ci_client.ci_client import CIClient
            self._logger = logging.getLogger('CIClientReceiver')
            self._mutex = threading.RLock()
            self._client = CIClient(
                logbroker_token=logbroker_token,
                topic=topic,
                source_id=source_id,
                check_id=check_id,
                check_type=check_type,
                number=number,
                partition=partition,
                task_id_string=task_id_string,
                sandbox_task_id=sandbox_task_id,
            )
        except Exception as e:
            logging.exception('Exception occurred while creating CI Client')
            raise Exception('Exception occurred while creating CI Client', e)
        logging.debug('successfully initiated CIClient')

    @property
    def partition_group(self):
        return self._client.partition_group

    def __call__(self, method, path, data):
        logging.debug('post to CIClient')
        try:
            if self.skip_results and 'results' in data:
                return
            with self._mutex:
                self._client.post_message(data)
        except Exception as e:
            logging.exception('Exception occurred while posting results to CI Client')
            raise Exception('Exception occurred while posting results to CI Client', e)
        logging.info('Successfully post to CIClient')

    def close(self):
        try:
            self._client.close()
        except Exception:
            logging.exception('Error in CI Client')
