import enum
import collections
import json
import logging
import os
import time
import traceback


class TaskInfoReport(object):
    def __init__(self, id, gsid, partition, stages):
        self.id = id
        self.gsid = gsid
        self.partition = partition
        self.stages = stages

    @staticmethod
    def create_task(task):
        return TaskInfoReport(
            task.task.id,
            task.context.GSID,
            task.options.partition_index,
            task.context.STAGES
        )


def get_payload(key, task, data):
    return json.dumps({
        'key': key,
        'timestamp': int(time.time()),
        'value': {
            'task_id': task.id,
            'gsid': task.gsid,
            'partition': task.partition,
            'payload': data,
        }
    })


def send_stages(proxy_task, logs_dir, unified_agent):
    stages_path = os.path.join(logs_dir, 'stages.json')
    if not os.path.exists(stages_path):
        logging.debug('Cannot send stages to logbroker: %s do not exist', stages_path)
        return

    try:
        with open(stages_path) as f:
            stages = json.load(f)
    except Exception:
        logging.debug('Fail to send stages to yt:\n%s', traceback.format_exc())
        payload = get_payload('stages', proxy_task, {'error': traceback.format_exc()})
    else:
        payload = get_payload('stages', proxy_task, stages)
    unified_agent.send(payload)


def get_distbs_worktime(proxy_task, graph_gen):
    DistbsWorktime = collections.namedtuple('DistbsWorktime', ['started', 'finished'])
    if graph_gen:
        return DistbsWorktime(proxy_task.stages['graph-distbs-worktime_started'], proxy_task.stages['graph-distbs-worktime_finished'])
    else:
        return DistbsWorktime(proxy_task.stages['distbs-worktime_started'], proxy_task.stages['distbs-worktime_finished'])


def send_distbs_overhead(proxy_task, logs_dir, unified_agent, graph_gen=False):
    key = 'gg_distbuild_overhead' if graph_gen else 'distbuild_overhead'
    distbs_result_path = os.path.join(logs_dir, 'distbs_result.json')
    if not os.path.exists(distbs_result_path):
        logging.debug('%s: Skip sending distbuild overhead to yt because of %s do not exist', key, distbs_result_path)
        return

    try:
        with open(distbs_result_path) as f:
            distbs_result = json.load(f)
    except Exception:
        logging.debug('%s: Fail to send disbuild overhead to yt:\n%s', key, traceback.format_exc())
        payload = get_payload(key, proxy_task, {'error': traceback.format_exc()})
        unified_agent.send(payload)
        return

    critical_path = distbs_result.get('critical_path')
    if not critical_path:
        logging.debug('%s: Skip sending distbuild overhead to yt because of missing/empty critical_path in %s', key, distbs_result_path)
        return

    critical_path = list(sorted(critical_path, key=lambda x: x['start_time']))
    critical_path_total_time = 0
    for node in critical_path:
        try:
            critical_path_total_time += node['finish_time'] - node['start_time']
        except KeyError as e:
            logging.debug('%s: Skip critical path node because of:\n%s', key, e)

    try:
        distbs_worktime = get_distbs_worktime(proxy_task, graph_gen)
        data = {
            'critical_path_total_time': critical_path_total_time,
            'distbs_overhead_total_time': int(distbs_worktime.finished - distbs_worktime.started) - critical_path_total_time,
            'distbs_overhead_inside_critical_path': critical_path[-1]['finish_time'] - critical_path[0]['start_time'] - critical_path_total_time,
            'distbs_overhead_before_first_node_execution': critical_path[0]['start_time'] - int(distbs_worktime.started),
            'distbs_overhead_after_last_node_execution': int(distbs_worktime.finished) - critical_path[-1]['finish_time']
        }
    except Exception:
        logging.debug('%s: Fail sending distbuild overhead to yt because of:\n%s', key, traceback.format_exc())
        payload = get_payload(key, proxy_task, {'error': traceback.format_exc()})
        unified_agent.send(payload)
        return

    payload = get_payload(key, proxy_task, data)
    unified_agent.send(payload)


class CacheMode(enum.Enum):
    FULL = 'full'
    MINIMAL = 'minimal'
    NONE = 'none'


def send_gg_cache_stats(proxy_task, logs_dir, unified_agent, cache_mode, gg_cache_is_found, autocheck_config_path, circuit_type, is_precommit, recheck):
    from yalibrary.ggaas.sandbox import get_gg_stat_from_log
    left_logs = os.path.join(logs_dir, 'left', 'generating.log')
    right_logs = os.path.join(logs_dir, 'right', 'generating.log')
    if not os.path.exists(left_logs) or not os.path.exists(right_logs):
        logging.debug('Cannot send graph generation cache statistics to logbroker: left or right log does not exist')
        return
    try:
        autocheck_config = 'UNKNOWN'
        if autocheck_config_path:
            autocheck_config = os.path.splitext(os.path.basename(autocheck_config_path))[0]
        cache_stats = {
            'left': get_gg_stat_from_log(left_logs),
            'right': get_gg_stat_from_log(right_logs),
            'is_fallback': cache_mode == CacheMode.NONE,  # For compatibility. The flag is put to the clickhouse table and used in Datalens charts
            'cache_mode': cache_mode.value,
            'gg_cache_is_found': gg_cache_is_found,
            'autocheck_config': autocheck_config,
            'circuit_type': circuit_type.lower() if circuit_type else 'UNKNOWN',
            'is_precommit': is_precommit,
            'recheck': recheck,
        }
    except Exception:
        logging.debug('Failed to send graph generation cache statistics to yt:\n%s', traceback.format_exc())
        payload = get_payload('gg_cache_stats', proxy_task, {'error': traceback.format_exc()})
    else:
        payload = get_payload('gg_cache_stats', proxy_task, cache_stats)
    unified_agent.send(payload)


def send_pessimized_by_nodes(proxy_task, affected_nodes, nodes_limit, unified_agent):
    data = {
        'reason': 'pessimized by nodes',
        'affected_nodes': affected_nodes,
        'limit': nodes_limit
    }
    payload = get_payload('pessimization', proxy_task, data)
    unified_agent.send(payload)
