# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib import yql_helpers as yqlh
from sandbox.projects.avia.lib.yt_helpers import copy_table_to_weak


YESTERDAY = datetime.today() - timedelta(days=1)

REDIRECT_ABT_METRICS_QUERY = '''
USE hahn;
PRAGMA yt.InferSchema = '1';  -- Table "logs/avia-redirect-error-log/1d" does not have any scheme attribute
PRAGMA yson.DisableStrict;

DECLARE $from_date AS Date;
DECLARE $to_date AS Date;
DECLARE $to_date_redir AS Date;
DECLARE $to_date_booking AS Date;
DECLARE $output_table AS String;

$FormatDate = DateTime::Format('%Y-%m-%d');
$from_date_str = $FormatDate($from_date);
$to_date_str = $FormatDate($to_date);
$to_date_redir_str = $FormatDate($to_date_redir);
$to_date_booking_str = $FormatDate($to_date_booking);
$from_date_str_1 = $FormatDate($from_date - Interval("P1D"));

$booking = (
SELECT *
FROM RANGE('home/avia/logs/avia-partner-booking-log', $from_date_str, $to_date_booking_str) with inline
);
$booking_partners = (select AGGREGATE_LIST(distinct billing_order_id) from $booking);

$today_reqids = (
select DISTINCT main_reqid as reqid
from RANGE('logs/avia-wizard-query-log/1d', $from_date_str, $to_date_str)
);

$redirects = (
select *
FROM RANGE('home/avia/logs/avia-redir-balance-by-day-log', $from_date_str, $to_date_redir_str) with inline
WHERE NATIONAL_VERSION == 'ru' and WIZARDREQID is not null
);

$partners = (
    select code, billing_order_id
    from `//home/rasp/reference/partner`
    where code is not null and billing_order_id is not null
);

$conversions = (
select dt,
    some(if(redirect_type=='direct_wizard', conversion, null)) as direct_wizard,
    some(if(redirect_type=='indirect_wizard', conversion, null)) as indirect_wizard
from range('//home/avia/logs/avia-conversion-log', $from_date_str_1, $to_date_booking_str)
group by substring(updated_at, 0, 10) as dt
);

$format = DateTime::Format("%Y-%m-%d");
$parse_dt = DateTime::Parse("%Y-%m-%d %H:%M:%S");
$dt_to_timestamp = ($dt) -> { return DateTime::ToSeconds(DateTime::MakeTimestamp($parse_dt($dt))); };

$wizards_with_redirects_with_booking = (
select
    reqid,
    yandexuid,
    serp_uuid,
    frod_filter,
    redir_money,
    seats,
    billing_order_id,
    marker,
    wizardRedirKey,
    booking_partner_redir,
    reqid_timestamp,
    redir_timestamp,
    partner_code,
    if(wizardRedirKey is not null, some(c.direct_wizard), some(c.indirect_wizard)) as conversion,
    some(c.dt) as c_dt,
    some($format(DateTime::FromSeconds(cast(r.UNIXTIME as Uint32)))) as r_dt,
    count(b.billing_order_id) as booking_count,
    sum(if(b.price>0, b.price,
            if(b.billing_order_id is not null, r.OFFER_PRICE*100.0/120.0, 0)
        )) as book_money,
    min($dt_to_timestamp(b.created_at)) as booking_timestamp
from $redirects as r
join $today_reqids as t on r.WIZARDREQID == t.reqid
left join (select * from $booking where status == 'paid') as b on b.marker=r.MARKER
left join $conversions as c on c.dt == $format(DateTime::FromSeconds(cast(r.UNIXTIME as Uint32)))
left join $partners as p on p.billing_order_id == r.BILLING_ORDER_ID
group by
    r.WIZARDREQID as reqid,
    cast(String::Substring(r.WIZARDREQID, 0, 10) as int64) as reqid_timestamp,
    cast(r.`TIMESTAMP` as int64) as redir_timestamp,
    r.YANDEXUID as yandexuid,
    r.SERP_UUID as serp_uuid,
    r.FILTER as frod_filter,
    r.PRICE_ECPC as redir_money,
    r.ADULT_SEATS + r.CHILDREN_SEATS as seats,
    r.BILLING_ORDER_ID as billing_order_id,
    r.MARKER as marker,
    (r.SEARCH_SHOW_ID ?? r.WIZARDREDIRKEY) as wizardRedirKey,
    ListHas($booking_partners, r.BILLING_ORDER_ID) as booking_partner_redir,
    p.code as partner_code
);

select * from $wizards_with_redirects_with_booking;

$shows = (
    select * from range('logs/avia-wizard-show-log/1d', $from_date_str, $to_date_redir_str)
);
$queries = (
select *
from RANGE('logs/avia-wizard-query-log/1d', $from_date_str, $to_date_redir_str)
);

$is_cpa = ($partner) -> {
    RETURN $partner in ('aeroflot', 'pobeda', 'utair') or ('dohop' <= $partner  and $partner < 'dohop_z');
};
$is_airline = ($partner) -> {
    RETURN $partner in ('s_seven', 'csa', 'rusline', 'azimuth');
};
$is_agent = ($partner) -> {
    RETURN not $is_cpa($partner) and not $is_airline($partner);
};

$redir_money_cpa = ($partner_code, $booking_count, $book_money, $redir_money) -> {
    return if($partner_code=='aeroflot', ($booking_count ?? 0) * 179.17,
           if($partner_code=='pobeda', ($book_money ?? 0) * 0.01 / 1.2,
           if($partner_code=='utair', if(($book_money ?? 0) == 0, 0,
                                      if($book_money <= 7000, 120.0 / 1.2,
                                      if($book_money <= 10000, 150.0 / 1.2, 170.0 / 1.2))),
               $redir_money)));
};

$wizards_with_redirects_with_booking_and_dated = (
    select
        r.reqid as reqid,
        r.yandexuid as yandexuid,
        r.serp_uuid as serp_uuid,
        q.icookie as icookie,
        q._rest as icookie_rest,
        r.frod_filter as frod_filter,
        sh.show_id as show_id,
        sh.job_id as job_id,
        sh.index as index,
        if(r.wizardRedirKey is not null, 1, 0) as straight,
        if(q._rest['raw_date'] is not null, 1, 0) as with_date_from_request,
        if(q.departure_date is not null, 1, 0) as with_departure_date,
        if(q.return_date is not null, 1, 0) as with_return_date,
        r.billing_order_id as billing_order_id,
        r.seats as seats,
        $redir_money_cpa(r.partner_code, r.booking_count, r.book_money, r.redir_money) as redir_money,
        r.redir_money as price_ecpc,
        r.booking_count ?? 0 as booking_count,
        r.book_money ?? 0 as booking_money,
        if(r.booking_partner_redir, 1, 0) as booking_partner_redir,
        r.partner_code as partner_code,
        if($is_agent(r.partner_code), 1, 0) as is_agent,
        r.conversion as conversion,
        $redir_money_cpa(r.partner_code, r.booking_count, r.book_money, r.redir_money)/r.conversion as redir_money_no_conversion,
        r.redir_timestamp-r.reqid_timestamp as time_from_request_to_redirect,
        if(r.booking_timestamp-r.redir_timestamp<0, 0, r.booking_timestamp-r.redir_timestamp) as time_from_redirect_to_booking,
        if(q.page=='search', 1, 0) as search_page
    from $wizards_with_redirects_with_booking as r
    left join $shows as sh on sh.show_id == r.wizardRedirKey
    left join $queries as q on q.job_id == sh.job_id
);

select * from $wizards_with_redirects_with_booking_and_dated;

$errors = (
select *
FROM RANGE('//home/avia/logs/avia-redirect-error-log', $from_date_str, $to_date_redir_str)
);

$get_user_id = ($serp_uuid, $yandexuid, $icookie) -> {
    RETURN 'y' || if($icookie is not null and $icookie!='' and len($icookie) < 30, $icookie, $yandexuid);
};

$make_parameter = ($metrica, $value) -> {
    RETURN if($value is null, '', '	'||$metrica||'='||cast($value as String));
};


INSERT INTO
    $output_table
WITH TRUNCATE
select * from (
select $get_user_id(serp_uuid, yandexuid, icookie) as key,
    ' stat_queries' as subkey,
    reqid ||
    $make_parameter('wiz_avia_fraud', frod_filter) ||
    $make_parameter('wiz_avia_redirect', 1) ||
    $make_parameter('wiz_avia_price', redir_money) ||
    $make_parameter('wiz_avia_price_ecpc', price_ecpc) ||
    $make_parameter('wiz_avia_straight_redirect', straight) ||
    $make_parameter('wiz_avia_index', index) ||
    $make_parameter('wiz_avia_with_date_from_request', with_date_from_request) ||
    $make_parameter('wiz_avia_with_departure_date', with_departure_date) ||
    $make_parameter('wiz_avia_with_return_date', with_return_date) ||
    $make_parameter('wiz_avia_seats', seats) ||
    $make_parameter('wiz_avia_time_from_request_to_redirect', time_from_request_to_redirect) ||
    $make_parameter('wiz_avia_partner', partner_code) ||
    $make_parameter('wiz_avia_is_agent', is_agent) ||
    $make_parameter('wiz_avia_conversion', conversion) ||
    $make_parameter('wiz_avia_redir_money_no_conversion', redir_money_no_conversion) ||
    $make_parameter('wiz_avia_booking_partner_redir', booking_partner_redir) ||
    $make_parameter('wiz_avia_search_page', search_page) ||
    if(booking_count==0, '',
        $make_parameter('wiz_avia_booking_count', booking_count) ||
        $make_parameter('wiz_avia_booking_money', booking_money) ||
        $make_parameter('wiz_avia_time_from_redirect_to_booking', time_from_redirect_to_booking)
    ) as value
from $wizards_with_redirects_with_booking_and_dated
where $get_user_id(serp_uuid, yandexuid, icookie) is not null

UNION ALL

select $get_user_id(serp_uuid, yandexuid, icookie) as key,
    ' stat_queries' as subkey,
    wizardReqId ||
    '	wiz_avia_fraud=0' ||
    '	wiz_avia_error=1' as value
from $errors
where $get_user_id(serp_uuid, yandexuid, icookie) is not null and wizardReqId is not null and wizardReqId != ''
)
where value is not null
order by key, subkey;
'''


class SendAviaMetricsToAbt(AviaBaseTask):
    """
    Send Avia metrics to ABT.
    See https://st.yandex-team.ru/USEREXP-5547 for metric definition
    """

    _yt_client = None
    _yql_client = None

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass  # We do not need caches

        environments = (
            PipEnvironment('yandex-yt', version='0.10.8'),
            PipEnvironment('requests'),
            PipEnvironment('yql')
        )

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Map reduce settings') as mr_block:
            mr_cluster = sdk2.parameters.String('MapReduce cluster', default='hahn', required=True)
            mr_user = sdk2.parameters.String('MapReduce user', required=True)
            mr_dir = sdk2.parameters.String('Directory', required=True, default='//home/avia/abt')
            yt_token_vault = sdk2.parameters.String('Token vault name', required=True, default='YT_TOKEN')

        with sdk2.parameters.Group('YQL settings') as yql_block:
            yql_token_vault = sdk2.parameters.String('YQL token vault name', required=True, default='YQL_TOKEN')

        with sdk2.parameters.Group('Date settings') as date_block:
            left_date = sdk2.parameters.String('Start date (default yesterday)', required=False)
            right_date = sdk2.parameters.String('End date (default yesterday)', required=False)

        with sdk2.parameters.Group('Debug settings') as debug_settings:
            debug_run = sdk2.parameters.Bool('Debug run', default=False, required=True)

    def _get_yt_client(self):
        if self._yt_client is None:
            import yt.wrapper as yt
            self._yt_client = yt.YtClient(
                proxy='hahn',
                token=self._get_token('yt'),
            )

        return self._yt_client

    def _get_yql_client(self):
        if self._yql_client is None:
            from yql.api.v1.client import YqlClient
            self._yql_client = YqlClient(
                token=self._get_token('yql'),
            )

        return self._yql_client

    def get_output_table(self, date):
        return os.path.join(self.Parameters.mr_dir, date.strftime('%Y%m%d'))

    def create_tag(self, date, cluster='hahn'):
        return 'cluster={cluster}_calc_avia_metrics_{date}'.format(
            cluster=cluster,
            date=date.strftime('%Y%m%d'),
        )

    def push_tag(self, tag):
        import requests
        return requests.get('http://rem-bernstein.n.yandex-team.ru/set_tag?tag=' + tag)

    def create_and_push_tag(self, date):
        tag = self.create_tag(date)
        logging.info('Tag: %s', tag)
        if not self.Parameters.debug_run:
            self.push_tag(tag)

    def process_date(self, date, output_path):
        # yandexuid -> metric_name -> value
        logging.info('Process date: %s', date.strftime('%Y-%m-%d'))

        temp_table = self.create_temp_table_with_abt_metrics(date)
        output_table = '{}/{}'.format(output_path, date.strftime('%Y%m%d'))
        copy_table_to_weak(self._get_yt_client(), from_table=temp_table, to_table=output_table, logger=logging)
        self.create_and_push_tag(date)

    def create_temp_table_with_abt_metrics(self, date):
        from yql.client.parameter_value_builder import YqlParameterValueBuilder as ValueBuilder

        yt_client = self._get_yt_client()
        yql_client = self._get_yql_client()

        temp_table = yt_client.create_temp_table()
        r = yql_client.query(REDIRECT_ABT_METRICS_QUERY,
                             syntax_version=1,
                             title='[YQL] Generating Avia wizard product metrics')
        r.run(parameters=ValueBuilder.build_json_map({
            '$from_date': ValueBuilder.make_date(date),
            '$to_date': ValueBuilder.make_date(date),
            '$to_date_redir': ValueBuilder.make_date(date + timedelta(days=1)),
            '$to_date_booking': ValueBuilder.make_date(date + timedelta(days=2)),
            '$output_table': ValueBuilder.make_string(temp_table),
        }))
        logging.info('YQL Operation: %s', yqlh.get_yql_operation_url(r))
        r.wait_progress()

        if not r.is_success:
            yqlh.log_errors(r, logging)
            raise SandboxTaskFailureError('YQL query failed')

        logging.info('YQL query done in %s table', temp_table)
        return temp_table

    def on_execute(self):
        logging.info('Start')

        left_date = self._parse_date(self.Parameters.left_date).date()
        logging.info('Left date: %s', left_date.strftime('%Y-%m-%d'))
        right_date = self._parse_date(self.Parameters.right_date).date()
        logging.info('Right date: %s', right_date.strftime('%Y-%m-%d'))

        yt_client = self._get_yt_client()
        if not yt_client.exists(self.Parameters.mr_dir):
            yt_client.create('map_node', self.Parameters.mr_dir, recursive=True)

        current_date = left_date
        while current_date <= right_date:
            self.process_date(current_date, self.Parameters.mr_dir)
            current_date += timedelta(days=1)

        logging.info('End')

    def _parse_date(self, s):
        if not s:
            return YESTERDAY
        try:
            return datetime.today() + timedelta(days=int(s))
        except ValueError:
            return datetime.strptime(s, '%Y-%m-%d')

    def _get_token(self, system):
        if system == 'yt':
            name = self.Parameters.yt_token_vault or 'YT_TOKEN'

        elif system == 'yql':
            name = self.Parameters.yql_token_vault or 'YQL_TOKEN'

        else:
            raise ValueError('Unknown system: {}'.format(system))

        return sdk2.Vault.data(self.Parameters.mr_user, name)
