# -*- encoding: utf-8 -*-
from __future__ import unicode_literals

import logging
from datetime import datetime

from sandbox import sdk2
from sandbox.common.types import resource as ctr
from sandbox.projects.common import binary_task, solomon

from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib.datetime_helpers import _dt_to_unixtime_utc, get_utc_now
from sandbox.projects.Travel.resources import dicts

log = logging.getLogger(__name__)


class AviaStatisticsUpdateCityToData(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        disk_space = 128  # 128 Megs or less
        ram = 128  # 128 Megs or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):

        SPECIFIC_DAY_KEY = 'specific_day'
        TODAY_KEY = 'today'

        # binary task release parameters
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Yt parameters') as yt_group:
            yt_proxy = sdk2.parameters.String('Proxy', default='hahn', required=True)
            yt_token = sdk2.parameters.YavSecret('YT token', default='sec-01dfxmszhq27tk66hm107d0ycd', required=True)

        with sdk2.parameters.Group('YQL parameters') as yql_group:
            yql_token = sdk2.parameters.YavSecret('YQL token', default='sec-01e4q88q2jcz84c8dtwrm29z3r', required=True)

        with sdk2.parameters.Group('YDB parameters') as ydb_group:
            ydb_cluster = sdk2.parameters.String('Cluster', default='ydb-ru-prestable.yandex.net:2135', required=True)
            ydb_database = sdk2.parameters.String(
                'Database',
                default='/ru-prestable/ticket/testing/avia_statistics',
                required=True,
            )
            ydb_token = sdk2.parameters.YavSecret('YDB token', default='sec-01e149a59chjhzr2c1f6wcqdef', required=True)
            ydb_batch_size = sdk2.parameters.Integer(
                'YDB batch size',
                required=True,
                default=1000,
            )
            ydb_city_to_route_crosslinks_table = sdk2.parameters.String(
                'City to route crosslinks table',
                default='city_to_route_crosslinks',
                required=True,
            )
            ydb_month_and_year_prices_table = sdk2.parameters.String(
                'Month and year prices by city to table',
                default='month_and_year_prices_by_city_to',
                required=True,
            )
            ydb_city_to_nearest_cities_table = sdk2.parameters.String(
                'Nearest cities table',
                default='city_to_nearest_cities',
                required=True,
            )

        with sdk2.parameters.Group('price-index parameters') as price_index:
            price_index_host = sdk2.parameters.String(
                'price-index host',
                required=True,
                default='http://price-index.production.avia.yandex.net',
            )
            price_index_timeout = sdk2.parameters.Integer(
                'price-index request timeout (seconds)',
                required=True,
                default=5,
            )
            window_size = sdk2.parameters.Integer(
                'Price searching window size in the future',
                required=True,
                default=30,
            )
            price_index_batch_size = sdk2.parameters.Integer(
                'price-index batch size',
                required=True,
                default=100,
            )

        with sdk2.parameters.Group('avia-backend parameters') as avia_backend:
            avia_backend_host = sdk2.parameters.String(
                'avia-backend host',
                required=True,
                default='http://backend.production.avia.yandex.net',
            )

        with sdk2.parameters.Group('Input YT tables') as yt_tables:
            yt_landing_cities_table = sdk2.parameters.String(
                'YT table with landing cities',
                default='//home/avia/avia-statistics/city-to-landing-cities',
                required=True,
            )
            yt_landing_routes_table = sdk2.parameters.String(
                'YT table with landing routes',
                default='//home/avia/avia-statistics/landing-routes',
                required=True,
            )
            yt_landing_route_weights_table = sdk2.parameters.String(
                'YT table with landing route weights',
                default='//home/avia/avia-statistics/landing-route-weights',
                required=True,
            )
            yt_redirects_log_directory = sdk2.parameters.String(
                'YT directory with avia redirects log',
                default='//home/avia/logs/avia-redir-balance-by-day-log',
                required=True,
            )

        with sdk2.parameters.Group('Nearest cities parameters') as nearest_cities_group:
            default_max_distance_km = sdk2.parameters.Integer('Default max distance km', default=200, required=True)

        with sdk2.parameters.RadioGroup('Run type') as log_type:
            log_type.values[TODAY_KEY] = log_type.Value('Run for today', default=True)
            log_type.values[SPECIFIC_DAY_KEY] = log_type.Value('Run for a specific day')

            with log_type.value[SPECIFIC_DAY_KEY]:
                date = sdk2.parameters.StrictString(
                    'Date YYYY-MM-DD',
                    regexp=r"\d{4}-\d{2}-\d{2}",
                    required=True,
                )

        with sdk2.parameters.Group('Solomon settings') as solomon_settings:
            solomon_project = sdk2.parameters.String('Solomon project', required=True, default='avia')
            solomon_cluster = sdk2.parameters.String('Solomon cluster', required=True, default='avia_statistics_testing')
            solomon_service = sdk2.parameters.String('Solomon service', required=True, default='city_to_landing')

    def on_execute(self):
        import ydb
        from yql.api.v1.client import YqlClient

        from travel.avia.avia_statistics.landing_cities import iter_landing_cities
        from travel.avia.avia_statistics.landing_routes import get_route_weights, iter_landing_routes
        from travel.avia.avia_statistics.services.price_index.client import Client as PriceIndexClient
        from travel.avia.avia_statistics.updaters.city_to.city_route_crosslinks_updater.lib.table import (
            CityRouteCrosslinksTable
        )
        from travel.avia.avia_statistics.updaters.city_to.city_route_crosslinks_updater.lib.updater import (
            CityRouteCrosslinksUpdater
        )
        from travel.avia.avia_statistics.updaters.city_to.month_and_year_prices_by_city_to_updater.lib.collector import (
            MonthAndYearPricesByCityToCollector
        )
        from travel.avia.avia_statistics.updaters.city_to.month_and_year_prices_by_city_to_updater.lib.table import (
            MonthAndYearPricesByCityToTable
        )
        from travel.avia.avia_statistics.updaters.city_to.month_and_year_prices_by_city_to_updater.lib.updater import (
            MonthAndYearPricesByCityToUpdater
        )
        from travel.avia.avia_statistics.updaters.city_to.nearest_cities_updater.lib.updater import CityToNearestCitiesUpdater
        from travel.avia.avia_statistics.updaters.city_to.nearest_cities_updater.lib.table import CityToNearestCitiesTable
        from travel.avia.library.python.backend_client.client import Client as BackendClient
        from travel.avia.library.python.lib_yt.client import configured_client
        from travel.avia.library.python.ydb.session_manager import YdbSessionManager
        from travel.library.python.dicts.avia.settlement_repository import SettlementRepository

        super(AviaStatisticsUpdateCityToData, self).on_execute()

        # Setup
        if self.Parameters.log_type == self.Parameters.TODAY_KEY:
            target_date = datetime.now().date()
        else:
            target_date = datetime.strptime(str(self.Parameters.date), '%Y-%m-%d').date()

        settlement_repository = SettlementRepository()
        settlement_resource_path = self._get_latest_resource_version_path(dicts.TRAVEL_DICT_AVIA_SETTLEMENTS_PROD)
        settlement_repository.load_from_file(settlement_resource_path)

        yt_client = configured_client(self.Parameters.yt_proxy, self.Parameters.yt_token.data()['token'])
        yql_client = YqlClient(token=self.Parameters.yql_token.data()['token'])

        ydb_config = ydb.DriverConfig(
            endpoint=self.Parameters.ydb_cluster,
            database=self.Parameters.ydb_database,
            auth_token=self.Parameters.ydb_token.data()['token']
        )
        ydb_session_manager = YdbSessionManager(ydb_config)

        price_index_client = PriceIndexClient(self.Parameters.price_index_host)
        backend_client = BackendClient(self.Parameters.avia_backend_host)

        landing_cities = list(iter_landing_cities(yt_client, self.Parameters.yt_landing_cities_table))
        landing_routes = list(iter_landing_routes(yt_client, self.Parameters.yt_landing_routes_table))[::-1]
        route_weights = get_route_weights(yt_client, self.Parameters.yt_landing_route_weights_table)

        # City to route crosslinks
        ydb_city_to_route_crosslinks_table = CityRouteCrosslinksTable(
            ydb_session_manager,
            self.Parameters.ydb_database,
            self.Parameters.ydb_city_to_route_crosslinks_table,
        )
        city_route_crosslinks_updater = CityRouteCrosslinksUpdater(
            yt_client=yt_client,
            landing_cities=landing_cities,
            landing_routes=landing_routes,
            route_weights=route_weights,
            settlement_repository=settlement_repository,
            city_route_crosslinks_table=ydb_city_to_route_crosslinks_table,
            avia_backend_client=backend_client,
            price_index_client=price_index_client,
            price_index_batch_size=self.Parameters.price_index_batch_size,
            price_index_timeout=self.Parameters.price_index_timeout,
            ydb_batch_size=self.Parameters.ydb_batch_size,
        )
        city_route_crosslinks_updater.update(
            target_date,
            window_size=self.Parameters.window_size,
        )

        # Month and year prices
        ydb_month_and_year_prices_table = MonthAndYearPricesByCityToTable(
            ydb_session_manager,
            self.Parameters.ydb_database,
            self.Parameters.ydb_month_and_year_prices_table,
        )
        month_and_year_prices_collector = MonthAndYearPricesByCityToCollector(
            yql_client=yql_client,
            redirects_log_path=self.Parameters.yt_redirects_log_directory,
            landing_cities_table=self.Parameters.yt_landing_cities_table,
        )
        month_and_year_prices_updater = MonthAndYearPricesByCityToUpdater(
            month_and_year_prices_collector,
            ydb_month_and_year_prices_table,
            self.Parameters.ydb_batch_size,
        )
        month_and_year_prices_updater.update(target_date)

        # Nearest cities
        ydb_city_to_nearest_cities_table = CityToNearestCitiesTable(
            ydb_session_manager,
            self.Parameters.ydb_database,
            self.Parameters.ydb_city_to_nearest_cities_table,
        )
        city_to_nearest_cities_updater = CityToNearestCitiesUpdater(
            yt_client=yt_client,
            landing_cities=landing_cities,
            city_to_nearest_cities_table=ydb_city_to_nearest_cities_table,
            avia_backend_client=backend_client,
            ydb_batch_size=self.Parameters.ydb_batch_size,
            default_max_distance_km=self.Parameters.default_max_distance_km,
        )
        city_to_nearest_cities_updater.update()

        # Send to solomon
        self.send_data_to_solomon(
            ydb_city_to_route_crosslinks_table=ydb_city_to_route_crosslinks_table,
            ydb_month_and_year_prices_table=ydb_month_and_year_prices_table,
            ydb_city_to_nearest_cities_table=ydb_city_to_nearest_cities_table,
        )

    def send_data_to_solomon(
            self,
            ydb_city_to_route_crosslinks_table,
            ydb_month_and_year_prices_table,
            ydb_city_to_nearest_cities_table,
    ):
        now = _dt_to_unixtime_utc(get_utc_now())

        shard_labels = {
            'project': self.Parameters.solomon_project,
            'cluster': self.Parameters.solomon_cluster,
            'service': self.Parameters.solomon_service,
        }
        log.info(shard_labels)
        sensors = [
            {
                'ts': now,
                'labels': {'sensor': 'route_crosslinks_row_count',
                           'table': self.Parameters.ydb_city_to_route_crosslinks_table},
                'value': ydb_city_to_route_crosslinks_table.count(),
            },
            {
                'ts': now,
                'labels': {'sensor': 'route_crosslinks_cities_count',
                           'table': self.Parameters.ydb_city_to_route_crosslinks_table},
                'value': ydb_city_to_route_crosslinks_table.cities_count(),
            },
            {
                'ts': now,
                'labels': {'sensor': 'month_and_year_prices_row_count',
                           'table': self.Parameters.ydb_month_and_year_prices_table},
                'value': ydb_month_and_year_prices_table.count(),
            },
            {
                'ts': now,
                'labels': {'sensor': 'nearest_cities_row_count',
                           'table': self.Parameters.ydb_city_to_nearest_cities_table},
                'value': ydb_city_to_nearest_cities_table.count(),
            },
        ]
        log.info('Sending sensors to solomon. %r', sensors)
        solomon.push_to_solomon_v2(self.solomon_token, shard_labels, sensors, common_labels=())

    def _get_latest_resource_version_path(self, resource_class):
        resource = resource_class.find(state=ctr.State.READY).first()
        resource_data = sdk2.ResourceData(resource)
        return resource_data.path.as_posix()
