# -*- encoding: utf-8 -*-
from __future__ import unicode_literals

import logging
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.projects.common import binary_task
from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.avia_statistics.update_top_airlines import AviaStatisticsUpdateTopAirlines

log = logging.getLogger(__name__)


class AviaStatisticsUpdateFlights(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        disk_space = 128  # 128 Megs or less
        ram = 128  # 128 Megs or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):
        SPECIFIC_DAY_KEY = 'specific_day'
        YESTERDAY_KEY = 'yesterday'

        # binary task release parameters
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Yt parameters'):
            yt_proxy = sdk2.parameters.String('Proxy', default='hahn', required=True)
            yt_token = sdk2.parameters.YavSecret('YT token', default='sec-01dfxmszhq27tk66hm107d0ycd', required=True)

        with sdk2.parameters.Group('YDB parameters'):
            ydb_cluster = sdk2.parameters.String('Cluster', default='ydb-ru-prestable.yandex.net:2135', required=True)
            ydb_database = sdk2.parameters.String('Database', default='/ru-prestable/ticket/testing/avia_statistics', required=True)
            ydb_table = sdk2.parameters.String('Table', default='flights', required=True)
            ydb_token = sdk2.parameters.YavSecret('YDB token', default='sec-01e149a59chjhzr2c1f6wcqdef', required=True)

        shared_flights_api_host = sdk2.parameters.String(
            'shared-flights api host',
            required=True,
            default='http://shared-flights.production.avia.yandex.net/api/v1',
        )
        variants_log_directory = sdk2.parameters.String(
            'avia-variants-log directory',
            required=True,
            default='//home/logfeller/logs/avia-variants-log/1d',
        )
        batch_size = sdk2.parameters.Integer(
            'Batch size',
            required=True,
            default=5000,
        )

        with sdk2.parameters.RadioGroup('Run type') as log_type:
            log_type.values[YESTERDAY_KEY] = log_type.Value('Run for yesterday', default=True)
            log_type.values[SPECIFIC_DAY_KEY] = log_type.Value('Run for a specific day')

            with log_type.value[SPECIFIC_DAY_KEY]:
                date = sdk2.parameters.StrictString(
                    'Date YYYY-MM-DD',
                    regexp="\d{4}-\d{2}-\d{2}",
                    required=True,
                )

    def on_execute(self):
        super(AviaStatisticsUpdateFlights, self).on_execute()
        from travel.avia.avia_statistics.flights_updater.lib.updater import FlightsUpdater
        from travel.avia.avia_statistics.flights_updater.lib.table import FlightsTable
        from travel.avia.avia_statistics.flights_updater.lib.collector import FlightsCollector
        from travel.avia.library.python.lib_yt.client import configured_client
        from travel.avia.library.python.ydb.session_manager import YdbSessionManager
        from travel.avia.library.python.references.station import create_station_cache
        from travel.avia.library.python.references.station_to_settlement import create_station_to_settlement_cache
        import ydb

        if self.Parameters.log_type == self.Parameters.YESTERDAY_KEY:
            target_date = (datetime.now() - timedelta(days=1)).date()
        else:
            target_date = datetime.strptime(str(self.Parameters.date), '%Y-%m-%d').date()

        yt_client = configured_client(self.Parameters.yt_proxy, self.Parameters.yt_token.data()['token'])

        ydb_config = ydb.DriverConfig(
            endpoint=self.Parameters.ydb_cluster,
            database=self.Parameters.ydb_database,
            auth_token=self.Parameters.ydb_token.data()['token']
        )

        session_manager = YdbSessionManager(ydb_config)
        flights_table = FlightsTable(session_manager, self.Parameters.ydb_database, self.Parameters.ydb_table)
        station_cache = create_station_cache(yt_client)
        station_to_settlement_cache = create_station_to_settlement_cache(yt_client)
        flights_collector = FlightsCollector(
            yt_client,
            self.Parameters.variants_log_directory,
            station_cache,
            station_to_settlement_cache,
        )
        flights_updater = FlightsUpdater(
            flights_collector,
            self.Parameters.shared_flights_api_host,
            flights_table,
            self.Parameters.batch_size,
        )
        flights_updater.update_flights(target_date)

        update_top_airlines_task_id = AviaStatisticsUpdateTopAirlines(
            self,
            yt_proxy=self.Parameters.yt_proxy,
            yt_token=self.Parameters.yt_token,
            ydb_cluster=self.Parameters.ydb_cluster,
            ydb_database=self.Parameters.ydb_database,
            ydb_token=self.Parameters.ydb_token,
            ydb_flights_table=self.Parameters.ydb_table,
        ).enqueue().id

        sdk2.WaitTask(
            [update_top_airlines_task_id],
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=True,
            timeout=60 * 60,  # 1 hour
        )
