# -*- encoding: utf-8 -*-
from __future__ import unicode_literals

import logging

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.common.types import resource as ctr
from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.Travel.resources import dicts

log = logging.getLogger(__name__)


class AviaStatisticsUpdateMinPricesByAirline(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        disk_space = 128  # 128 Megs or less
        ram = 128  # 128 Megs or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):

        # binary task release parameters
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Yt parameters'):
            yt_proxy = sdk2.parameters.String('Proxy', default='hahn', required=True)
            yt_token = sdk2.parameters.YavSecret('YT token', default='sec-01dfxmszhq27tk66hm107d0ycd', required=True)

        yql_token = sdk2.parameters.YavSecret('YQL token', default='sec-01e4q88q2jcz84c8dtwrm29z3r', required=True)

        with sdk2.parameters.Group('YDB parameters'):
            ydb_cluster = sdk2.parameters.String('Cluster', default='ydb-ru-prestable.yandex.net:2135', required=True)
            ydb_database = sdk2.parameters.String('Database', default='/ru-prestable/ticket/testing/avia_statistics', required=True)
            ydb_table = sdk2.parameters.String('Table', default='min_prices_by_airline', required=True)
            ydb_token = sdk2.parameters.YavSecret('YDB token', default='sec-01e149a59chjhzr2c1f6wcqdef', required=True)

        batch_size = sdk2.parameters.Integer(
            'Batch size',
            required=True,
            default=1000,
        )
        landing_routes_table = sdk2.parameters.String('YT table with landing routes', default='//home/avia/avia-statistics/landing-routes', required=True)

    def on_execute(self):
        super(AviaStatisticsUpdateMinPricesByAirline, self).on_execute()
        from travel.avia.avia_statistics.min_prices_by_airline_updater.lib.collector import MinPricesByAirlineCollector
        from travel.avia.avia_statistics.min_prices_by_airline_updater.lib.updater import MinPricesByAirlineUpdater
        from travel.avia.avia_statistics.min_prices_by_airline_updater.lib.table import MinPricesByAirlineTable
        from travel.avia.avia_statistics.landing_routes import get_landing_routes
        from travel.avia.library.python.lib_yt.client import configured_client
        from travel.avia.library.python.ydb.session_manager import YdbSessionManager
        from travel.library.python.dicts.avia.company_repository import CompanyRepository
        from yql.api.v1.client import YqlClient
        import ydb

        yql_client = YqlClient(token=self.Parameters.yql_token.data()['token'])

        yt_client = configured_client(self.Parameters.yt_proxy, self.Parameters.yt_token.data()['token'])
        landing_routes = get_landing_routes(yt_client, self.Parameters.landing_routes_table)

        ydb_config = ydb.DriverConfig(
            endpoint=self.Parameters.ydb_cluster,
            database=self.Parameters.ydb_database,
            auth_token=self.Parameters.ydb_token.data()['token']
        )
        session_manager = YdbSessionManager(ydb_config)

        company_repository = CompanyRepository()
        company_resource_path = self._get_latest_resource_version_path(dicts.TRAVEL_DICT_AVIA_COMPANY_PROD)
        company_repository.load_from_file(company_resource_path)

        min_prices_by_airline_collector = MinPricesByAirlineCollector(yql_client, company_repository)
        min_prices_by_airline_table = MinPricesByAirlineTable(session_manager, self.Parameters.ydb_database, self.Parameters.ydb_table)
        min_prices_by_airline_updater = MinPricesByAirlineUpdater(
            landing_routes,
            min_prices_by_airline_collector,
            min_prices_by_airline_table,
            self.Parameters.batch_size,
        )
        min_prices_by_airline_updater.update()

    @staticmethod
    def _get_latest_resource_version_path(resource_class):
        resource = resource_class.find(state=ctr.State.READY).first()
        resource_data = sdk2.ResourceData(resource)
        return resource_data.path.as_posix()
