# -*- encoding: utf-8 -*-
from __future__ import unicode_literals

import logging
from datetime import datetime

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.avia.base import AviaBaseTask

log = logging.getLogger(__name__)


class AviaStatisticsUpdatePopularMonths(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        disk_space = 128  # 128 Megs or less
        ram = 128  # 128 Megs or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):
        SPECIFIC_DAY_KEY = 'specific_day'
        TODAY_KEY = 'today'

        # binary task release parameters
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Yt parameters'):
            yt_proxy = sdk2.parameters.String('Proxy', default='hahn', required=True)
            yt_token = sdk2.parameters.YavSecret('YT token', default='sec-01dfxmszhq27tk66hm107d0ycd', required=True)

        yql_token = sdk2.parameters.YavSecret('YQL token', default='sec-01e4q88q2jcz84c8dtwrm29z3r', required=True)

        with sdk2.parameters.Group('YDB parameters'):
            ydb_cluster = sdk2.parameters.String('Cluster', default='ydb-ru-prestable.yandex.net:2135', required=True)
            ydb_database = sdk2.parameters.String('Database', default='/ru-prestable/ticket/testing/avia_statistics', required=True)
            ydb_table = sdk2.parameters.String('Table', default='popular_months', required=True)
            ydb_token = sdk2.parameters.YavSecret('YDB token', default='sec-01e149a59chjhzr2c1f6wcqdef', required=True)

        redirects_log_directory = sdk2.parameters.String(
            'avia-json-redir-log directory',
            required=True,
            default='//home/avia/logs/avia-redir-balance-by-day-log',
        )
        batch_size = sdk2.parameters.Integer(
            'Batch size',
            required=True,
            default=1000,
        )

        with sdk2.parameters.RadioGroup('Run type') as log_type:
            log_type.values[TODAY_KEY] = log_type.Value('Run for today', default=True)
            log_type.values[SPECIFIC_DAY_KEY] = log_type.Value('Run for a specific day')

            with log_type.value[SPECIFIC_DAY_KEY]:
                date = sdk2.parameters.StrictString(
                    'Date YYYY-MM-DD',
                    regexp="\d{4}-\d{2}-\d{2}",
                    required=True,
                )

    def on_execute(self):
        super(AviaStatisticsUpdatePopularMonths, self).on_execute()
        from travel.avia.avia_statistics.popular_months_by_route_updater.lib.updater import PopularMonthsWithPriceUpdater
        from travel.avia.avia_statistics.popular_months_by_route_updater.lib.table import PopularMonthWithPriceTable
        from travel.avia.avia_statistics.popular_months_by_route_updater.lib.collector import PopularMonthsCollector
        from travel.avia.library.python.lib_yt.client import configured_client
        from travel.avia.library.python.ydb.session_manager import YdbSessionManager
        from travel.avia.library.python.references.station import create_station_cache
        from travel.avia.library.python.references.station_to_settlement import create_station_to_settlement_cache

        import ydb
        from yql.api.v1.client import YqlClient

        if self.Parameters.log_type == self.Parameters.TODAY_KEY:
            target_date = datetime.now().date()
        else:
            target_date = datetime.strptime(str(self.Parameters.date), '%Y-%m-%d').date()

        yt_client = configured_client(self.Parameters.yt_proxy, self.Parameters.yt_token.data()['token'])
        yql_client = YqlClient(token=self.Parameters.yql_token.data()['token'])

        ydb_config = ydb.DriverConfig(
            endpoint=self.Parameters.ydb_cluster,
            database=self.Parameters.ydb_database,
            auth_token=self.Parameters.ydb_token.data()['token']
        )

        session_manager = YdbSessionManager(ydb_config)
        popular_months_table = PopularMonthWithPriceTable(
            session_manager,
            self.Parameters.ydb_database,
            self.Parameters.ydb_table,
        )
        station_cache = create_station_cache(yt_client)
        station_to_settlement_cache = create_station_to_settlement_cache(yt_client)
        popular_months_collector = PopularMonthsCollector(
            yql_client,
            self.Parameters.redirects_log_directory,
            station_cache,
            station_to_settlement_cache,
        )
        popular_months_updater = PopularMonthsWithPriceUpdater(
            popular_months_collector,
            popular_months_table,
            self.Parameters.batch_size,
        )
        popular_months_updater.update_popular_months(target_date)
