# -*- encoding: utf-8 -*-
from __future__ import unicode_literals

import logging
from datetime import datetime

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.avia.base import AviaBaseTask

log = logging.getLogger(__name__)


class AviaStatisticsUpdateReturnTicketPrices(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        disk_space = 128  # 128 Megs or less
        ram = 128  # 128 Megs or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):
        SPECIFIC_DAY_KEY = 'specific_day'
        TODAY_KEY = 'today'

        # binary task release parameters
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Yt parameters'):
            yt_proxy = sdk2.parameters.String('Proxy', default='hahn', required=True)
            yt_token = sdk2.parameters.YavSecret('YT token', default='sec-01dfxmszhq27tk66hm107d0ycd', required=True)

        with sdk2.parameters.Group('YDB parameters'):
            ydb_cluster = sdk2.parameters.String('Cluster', default='ydb-ru-prestable.yandex.net:2135', required=True)
            ydb_database = sdk2.parameters.String('Database', default='/ru-prestable/ticket/testing/avia_statistics', required=True)
            ydb_table = sdk2.parameters.String('Table', default='return_ticket_prices', required=True)
            ydb_token = sdk2.parameters.YavSecret('YDB token', default='sec-01e149a59chjhzr2c1f6wcqdef', required=True)

        price_index_host = sdk2.parameters.String(
            'price-index host',
            required=True,
            default='http://price-index.production.avia.yandex.net',
        )
        price_index_timeout = sdk2.parameters.Integer(
            'price-index request timeout (seconds)',
            required=True,
            default=5,
        )
        redirects_log_directory = sdk2.parameters.String(
            'avia-json-redir-log directory',
            required=True,
            default='//home/avia/logs/avia-json-redir-log',
        )
        batch_size = sdk2.parameters.Integer(
            'Batch size',
            required=True,
            default=100,
        )
        window_size = sdk2.parameters.Integer(
            'Price searching window size in the future',
            required=True,
            default=30,
        )
        landing_routes_table = sdk2.parameters.String('YT table with landing routes', default='//home/avia/avia-statistics/landing-routes', required=True)

        with sdk2.parameters.RadioGroup('Run type') as log_type:
            log_type.values[TODAY_KEY] = log_type.Value('Run for today', default=True)
            log_type.values[SPECIFIC_DAY_KEY] = log_type.Value('Run for a specific day')

            with log_type.value[SPECIFIC_DAY_KEY]:
                date = sdk2.parameters.StrictString(
                    'Date YYYY-MM-DD',
                    regexp="\d{4}-\d{2}-\d{2}",
                    required=True,
                )

    def on_execute(self):
        super(AviaStatisticsUpdateReturnTicketPrices, self).on_execute()
        from travel.avia.avia_statistics.return_ticket_prices_updater.lib.updater import ReturnTicketPricesUpdater
        from travel.avia.avia_statistics.return_ticket_prices_updater.lib.table import ReturnTicketPriceTable
        from travel.avia.avia_statistics.landing_routes import get_landing_routes
        from travel.avia.avia_statistics.services.price_index.client import Client as PriceIndexClient
        from travel.avia.library.python.lib_yt.client import configured_client
        from travel.avia.library.python.ydb.session_manager import YdbSessionManager
        import ydb

        if self.Parameters.log_type == self.Parameters.TODAY_KEY:
            target_date = datetime.now().date()
        else:
            target_date = datetime.strptime(str(self.Parameters.date), '%Y-%m-%d').date()

        yt_client = configured_client(self.Parameters.yt_proxy, self.Parameters.yt_token.data()['token'])

        ydb_config = ydb.DriverConfig(
            endpoint=self.Parameters.ydb_cluster,
            database=self.Parameters.ydb_database,
            auth_token=self.Parameters.ydb_token.data()['token']
        )

        session_manager = YdbSessionManager(ydb_config)
        return_ticket_price_table = ReturnTicketPriceTable(
            session_manager,
            self.Parameters.ydb_database,
            self.Parameters.ydb_table,
        )
        landing_routes = get_landing_routes(yt_client, self.Parameters.landing_routes_table)
        price_index_client = PriceIndexClient(self.Parameters.price_index_host)
        return_ticket_price_updater = ReturnTicketPricesUpdater(
            landing_routes,
            return_ticket_price_table,
            price_index_client,
            self.Parameters.batch_size,
            self.Parameters.price_index_timeout,
        )
        return_ticket_price_updater.update_return_ticket_prices(target_date, window_size=self.Parameters.window_size)
