# -*- encoding: utf-8 -*-
from __future__ import unicode_literals

import logging

from sandbox import sdk2
from sandbox.common.types import resource as ctr
from sandbox.projects.common import binary_task
from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia_resources.resources import AVIA_WIZARD_DIRECTION_NATIONAL
from sandbox.projects.Travel.resources import dicts

log = logging.getLogger(__name__)


class AviaStatisticsUpdateRouteInfos(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        disk_space = 128  # 128 Megs or less
        ram = 128  # 128 Megs or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):

        # binary task release parameters
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Yt parameters'):
            yt_proxy = sdk2.parameters.String('Proxy', default='hahn', required=True)
            yt_token = sdk2.parameters.YavSecret('YT token', default='sec-01dfxmszhq27tk66hm107d0ycd', required=True)

        with sdk2.parameters.Group('YDB parameters'):
            ydb_cluster = sdk2.parameters.String('Cluster', default='ydb-ru-prestable.yandex.net:2135', required=True)
            ydb_database = sdk2.parameters.String('Database', default='/ru-prestable/ticket/testing/avia_statistics', required=True)
            ydb_table = sdk2.parameters.String('Table', default='route_infos', required=True)
            ydb_token = sdk2.parameters.YavSecret('YDB token', default='sec-01e149a59chjhzr2c1f6wcqdef', required=True)

        shared_flights_api_host = sdk2.parameters.String(
            'shared-flights api host',
            required=True,
            default='http://shared-flights.testing.avia.yandex.net',
        )

        batch_size = sdk2.parameters.Integer(
            'Batch size',
            required=True,
            default=1000,
        )
        landing_routes_table = sdk2.parameters.String('YT table with landing routes', default='//home/avia/avia-statistics/landing-routes', required=True)

    def on_execute(self):
        super(AviaStatisticsUpdateRouteInfos, self).on_execute()
        import ydb

        from travel.avia.library.python.lib_yt.client import configured_client
        from travel.avia.library.python.ydb.session_manager import YdbSessionManager
        from travel.library.python.dicts.avia.direction_national_repository import DirectionNationalRepository
        from travel.library.python.dicts.avia.settlement_repository import SettlementRepository
        from travel.library.python.dicts.avia.station_repository import StationRepository
        from travel.avia.avia_statistics.route_infos_updater.lib.updater import RouteInfosUpdater
        from travel.avia.avia_statistics.route_infos_updater.lib.table import RouteInfosTable
        from travel.avia.avia_statistics.lib.settlements_geo_index import SettlementsGeoIndex
        from travel.avia.avia_statistics.landing_routes import get_landing_routes
        from travel.avia.library.python.shared_flights_client.client import SharedFlightsClient
        from travel.avia.library.python.references.station_to_settlement import create_station_to_settlement_cache
        from travel.avia.library.python.references.station import create_station_cache

        settlement_repository = SettlementRepository()
        settlement_resource_path = self._get_latest_resource_version_path(dicts.TRAVEL_DICT_AVIA_SETTLEMENTS_PROD)
        settlement_repository.load_from_file(settlement_resource_path)

        station_repository = StationRepository()
        station_resource_path = self._get_latest_resource_version_path(dicts.TRAVEL_DICT_AVIA_STATION_PROD)
        station_repository.load_from_file(station_resource_path)

        direction_national_repository = DirectionNationalRepository()
        direction_national_resource_path = self._get_latest_resource_version_path(AVIA_WIZARD_DIRECTION_NATIONAL)
        direction_national_repository.load_from_file(direction_national_resource_path)

        settlements_geo_index = SettlementsGeoIndex(direction_national_repository, settlement_repository)

        yt_client = configured_client(self.Parameters.yt_proxy, self.Parameters.yt_token.data()['token'])

        ydb_config = ydb.DriverConfig(
            endpoint=self.Parameters.ydb_cluster,
            database=self.Parameters.ydb_database,
            auth_token=self.Parameters.ydb_token.data()['token']
        )

        session_manager = YdbSessionManager(ydb_config)
        route_infos_table = RouteInfosTable(
            session_manager,
            self.Parameters.ydb_database,
            self.Parameters.ydb_table,
        )
        landing_routes = get_landing_routes(yt_client, self.Parameters.landing_routes_table)
        station_to_settlement_cache = create_station_to_settlement_cache(yt_client)
        station_cache = create_station_cache(yt_client)
        shared_flights_client = SharedFlightsClient(self.Parameters.shared_flights_api_host)
        route_infos_updater = RouteInfosUpdater(
            landing_routes,
            station_repository,
            station_to_settlement_cache,
            station_cache,
            settlements_geo_index,
            route_infos_table,
            shared_flights_client,
            self.Parameters.batch_size,
        )
        route_infos_updater.update()

    def _get_latest_resource_version_path(self, resource_class):
        resource = resource_class.find(state=ctr.State.READY).first()
        resource_data = sdk2.ResourceData(resource)
        return resource_data.path.as_posix()
