# coding: utf-8
import logging
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib.logs import configure_logging, get_sentry_dsn
from sandbox.projects.avia.lib.yt_helpers import YtClientFactory, tables_for_daterange


def calculate_flights(yt_client, destination_table, period):
    yesterday = (datetime.now() - timedelta(days=1)).date()
    source_tables = tables_for_daterange(
        yt_client,
        '//logs/avia-wizard-query-log/1d',
        yesterday - timedelta(days=period),
        yesterday,
    )

    def map_flight_number(input_row):
        flight_number = input_row.get('flight_number')
        if not flight_number:
            flight_number = input_row.get('_rest', {}).get('flight_number')

        if flight_number:
            yield {'flight_number': flight_number}

    def count_flights(key, rows):
        flight_number = key['flight_number']

        amount = 0
        for row in rows:
            amount += 1

        yield {'flight_number': flight_number, 'popularity': amount, '_sort_key': -amount}

    with yt_client.Transaction():
        yt_client.run_map_reduce(
            source_table=source_tables,
            destination_table=destination_table,
            mapper=map_flight_number,
            reducer=count_flights,
            reduce_by=['flight_number'],
        )

        yt_client.run_sort(
            source_table=destination_table,
            sort_by=['_sort_key'],
        )


class AviaCalculateWizardFlights(AviaBaseTask):
    """ Calculating statisitcsc of flight wizard """

    class Requirements(sdk2.Task.Requirements):
        environments = (
            PipEnvironment('raven'),
            PipEnvironment('yandex-yt', version='0.10.8'),
            PipEnvironment('yandex-yt-yson-bindings-skynet', version='0.3.32-0'),
        )

        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        reference_yt_user = 'robot-avia-api-pilot'
        reference_yt_proxy = 'hahn'
        reference_yt_path = '//home/rasp/reference'

        with sdk2.parameters.Group('Task settings') as task_settings:
            period = sdk2.parameters.Integer('Number of days', required=True, default=90)
            destination_table = sdk2.parameters.String(
                'Destination table',
                default='//home/avia/reports/wizard/popular_flights',
            )

        with sdk2.parameters.Group('YT Settings') as yt_settings:
            vaults_owner = sdk2.parameters.String('Token vault owner', required=True)
            yt_vault_name = sdk2.parameters.String('YT Token vault name', required=True, default='YT_TOKEN')
            yt_proxy = sdk2.parameters.String('YT cluster', required=True, default='hahn')

    def on_execute(self):
        configure_logging(get_sentry_dsn(self))
        logging.info('Start')
        yt_client = YtClientFactory().create(
            proxy=self.Parameters.yt_proxy,
            token=sdk2.Vault.data(self.Parameters.vaults_owner, self.Parameters.yt_vault_name),
        )

        logging.info('Start work')
        calculate_flights(yt_client, self.Parameters.destination_table, self.Parameters.period)
        logging.info('End')
