# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
from collections import namedtuple
from datetime import datetime, timedelta

import sandbox.sandboxsdk.environments as sdk_environments
import six
from sandbox import sdk2

from sandbox.projects.avia.base import AviaBaseTask

logger = logging.getLogger(__name__)

MAX_UPDATED_AT_YQL_TEMPLATE = """
USE `hahn`;
SELECT
    orders.billing_order_id,
    orders.partner_name,
    DateTime::Format('%Y-%m-%d')(CAST(MAX(orders.updated_at) as DateTime)) as max_updated_at
FROM
    `{cpa_order_path}` as orders
    JOIN `//home/rasp/reference/partner` as partner
    ON orders.billing_order_id = partner.billing_order_id
WHERE partner.use_in_update_conversions = true
GROUP BY orders.billing_order_id, orders.partner_name;
""".strip()


class RunYqlException(Exception):
    pass


@six.python_2_unicode_compatible
class PartnerInfo(namedtuple('PartnerInfo', 'billing_order_id, partner_name, updated_at')):
    def __str__(self):
        return six.text_type('Partner "{}" with billing_order_id={} updated_at={}').format(
            self.partner_name, self.billing_order_id, self.updated_at)


def _parse_date(dt):
    return datetime.strptime(dt, '%Y-%m-%d').date() if dt else None


class CheckUpdatedAt(AviaBaseTask):
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

        environments = [
            sdk_environments.PipEnvironment('six'),
            sdk_environments.PipEnvironment('yandex-yt', version='0.10.8'),
            sdk_environments.PipEnvironment('yql'),
        ]

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('YQL settings') as yql_block:
            yql_token_secret = sdk2.parameters.YavSecret('YQL token secret', required=True,
                                                         default='sec-01e4q88q2jcz84c8dtwrm29z3r')
            yql_token_key = sdk2.parameters.String('YQL token key', required=True, default='token')
        with sdk2.parameters.Group('Check parameters') as check_parameters:
            alert_threshold = sdk2.parameters.Integer('Alert threshold (days)', required=True, default=2)
            cpa_order_yt_path = sdk2.parameters.String('Path to CPA orders in YT', required=True,
                                                       default='//home/travel/prod/cpa/avia/orders')

    def on_execute(self):
        from yql.api.v1.client import YqlClient
        from sandbox.projects.avia.lib import yql_helpers

        logger.info('Start to check updated_at')

        yql_token = self.Parameters.yql_token_secret.data()[self.Parameters.yql_token_key]
        yql_client = YqlClient(token=yql_token)
        max_updated_at_yql = MAX_UPDATED_AT_YQL_TEMPLATE.format(cpa_order_path=self.Parameters.cpa_order_yt_path)
        yql_query = yql_client.query(max_updated_at_yql, syntax_version=1)
        yql_query.run()
        yql_query.wait_progress()

        if not yql_query.is_ok:
            yql_helpers.log_errors(yql_query, logger)
            raise RunYqlException('Error while running YQL')

        logger.info('YQL query is done')
        today = datetime.utcnow().date()
        suspicious_partners = []
        for table in yql_query.get_results():
            table.fetch_full_data()
            for billing_order_id, partner_name, max_updated_at_str in table.rows:
                max_updated_at = _parse_date(max_updated_at_str)
                if today - max_updated_at >= timedelta(days=self.Parameters.alert_threshold):
                    suspicious_partners.append(PartnerInfo(billing_order_id, partner_name, max_updated_at_str))

        if suspicious_partners:
            logger.error('We have %s suspicious partners', len(suspicious_partners))
            raise Exception('Updated_at is too old for\n\t{}'.format('\n\t'.join(
                (six.text_type(p) for p in suspicious_partners)
            )))
        else:
            logger.info('Nothing to worry about')
