import logging

from sandbox import sdk2
from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib.logs import configure_logging, get_sentry_dsn
from sandbox.projects.common import binary_task
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


class AviaCountryRestrictionsBaseTask(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    logger = None

    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Parameters') as executable_params:
            base_path = sdk2.parameters.String(
                'Base YT path',
                required=True,
                default='//home/avia/data/country-restrictions',
            )
            yt_proxies = sdk2.parameters.List(label='Yt proxies', required=True, default=['hahn', 'arnold'])

            with sdk2.parameters.RadioGroup('Environment') as environment:
                environment.values.TESTING = environment.Value('testing', default=True)
                environment.values.PRODUCTION = environment.Value('production')

        with sdk2.parameters.Group('Secrets') as secrets_settings:
            yt_token_secret = sdk2.parameters.YavSecret(
                'Yav secret with YT token',
                required=True,
                default='sec-01dfxmszhq27tk66hm107d0ycd',
            )
            solomon_token_secret = sdk2.parameters.YavSecret(
                'Yav secret with Solomon token',
                required=True,
                default='sec-01dwy9p37p1e8j94c5kp9zqmw1',
            )
            sandbox_token_secret = sdk2.parameters.YavSecret(
                'Yav secret with Sandbox token',
                required=True,
                default='sec-01ffs98ww23vytzg2r52020ngb',
            )

    def on_execute(self):
        self.logger.info('Execution started')
        super(AviaCountryRestrictionsBaseTask, self).on_execute()
        parser = None

        try:
            from travel.avia.country_restrictions.lib.types import Environment
            from travel.avia.country_restrictions.lib.kwargs_prepare import kwargs_prepare

            env = 'unknown'
            if self.Parameters.environment == 'PRODUCTION':
                env = 'production'
            elif self.Parameters.environment == 'TESTING':
                env = 'testing'

            parser = self.parser_class(**kwargs_prepare(
                yt_token=self.get_secret_value(self.Parameters.yt_token_secret, 'token'),
                sandbox_token=self.get_secret_value(self.Parameters.sandbox_token_secret, 'token'),
                solomon_token=self.get_secret_value(self.Parameters.solomon_token_secret, 'oauth'),
                version=1,
                base_path=str(self.Parameters.base_path),
                yt_proxies=self.Parameters.yt_proxies,
                environment=Environment.from_str(env),
            ))
            self.logger.info('Got country restrictions parser with name {}'.format(parser.PARSER_NAME))
        except Exception as e:
            raise SandboxTaskFailureError('Unable to get country restrictions parser: {}'.format(e))

        try:
            self.logger.info('Launching country restrictions parser {}'.format(parser.PARSER_NAME))
            parser.run()
            self.logger.info('Task has been completed.')
        except Exception as e:
            raise SandboxTaskFailureError(
                'Unable to run country restrictions parser {}. {}'.format(parser.PARSER_NAME, e),
            )

    def on_prepare(self):
        configure_logging(
            sentry_dsn=get_sentry_dsn(self)
        )
        self.logger = logging.getLogger(__name__)

        super(AviaCountryRestrictionsBaseTask, self).on_prepare()
