# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib.yt_helpers import YtClientFactory

YESTERDAY = datetime.today() - timedelta(days=1)


class TravelCreateYtSymlink(AviaBaseTask):
    """
    Make a yt-symlink from date-separated links to one big cpa-table
    For cofe-metrics TRAVELANALYTICS-176
    """

    _yt_client = None

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass  # We do not need caches

        environments = (
            PipEnvironment('yandex-yt', version='0.10.8'),
            PipEnvironment('yandex-yt-yson-bindings-skynet', version='0.3.32-0'),
        )

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('YT settings') as mr_block:
            cluster = sdk2.parameters.String('YT cluster', default='hahn', required=True)
            vaults_owner = sdk2.parameters.String('YT user', required=True)
            yt_token_vault_name = sdk2.parameters.String('YT Token vault name', required=True, default='YT_TOKEN')
            yt_path = sdk2.parameters.String('Table path', required=True, default='//home/travel/prod/cpa/train')
            yt_dir = sdk2.parameters.String('Directory', required=True, default='//home/avia/dev/kateov/train_cpa_links')

        with sdk2.parameters.Group('Date settings') as date_block:
            date = sdk2.parameters.String('Signal date (default yesterday, -2 - the day before yesterday)',
                                          required=False)

        with sdk2.parameters.Group('Debug settings') as debug_settings:
            debug_run = sdk2.parameters.Bool('Debug run', default=False, required=True)

    @property
    def yt_client(self):
        if self._yt_client is None:
            self._yt_client = YtClientFactory.create(
                proxy=self.Parameters.cluster,
                token=sdk2.Vault.data(self.Parameters.vaults_owner,
                                      self.Parameters.yt_token_vault_name),
            )

        return self._yt_client

    def create_symlink(self, date, yt_path, yt_dir):
        if not self.yt_client.exists(self.Parameters.yt_dir):
            self.yt_client.create('map_node', self.Parameters.yt_dir, recursive=True)

        yt_link_path = '{}/{}'.format(yt_dir, date.strftime('%Y-%m-%d'))
        logging.info('Symlink: %s', yt_link_path)

        self.yt_client.link(yt_path, yt_link_path, force=True)

    def on_execute(self):
        logging.info('Start')

        date = self._parse_date(self.Parameters.date).date()
        logging.info('Signal date: %s', date.strftime('%Y-%m-%d'))

        self.create_symlink(date=date,
                            yt_path=self.Parameters.yt_path,
                            yt_dir=self.Parameters.yt_dir)

        logging.info('End')

    def _parse_date(self, s):
        if not s:
            return YESTERDAY
        try:
            return datetime.today() + timedelta(days=int(s))
        except ValueError:
            return datetime.strptime(s, '%Y-%m-%d')
