# coding=utf-8

import logging
import re
from copy import deepcopy
from typing import Dict, List, Any

import sandbox.projects.common.build.parameters as build_parameters
from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.projects.rasp.qloud.api import QloudPrivateApi
from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib.qloud.api import QloudPublicApi

QLOUD_API_TIMEOUT = 120


class AviaDeployStand(AviaBaseTask):
    class Parameters(sdk2.Task.Parameters):
        arcadia_url = build_parameters.ArcadiaUrl()
        arcadia_patch = build_parameters.ArcadiaPatch()

        with sdk2.parameters.Group('Project') as project_params:
            application_title = sdk2.parameters.String(
                'Application title',
                description='This title will be used in logs, description, etc.',
                required=True,
                default='wizard',
            )

        with sdk2.parameters.Group('Qloud parameters') as qloud_params:
            qloud_token_vault_name = sdk2.parameters.String(
                'Vault secret with qloud oauth token',
                required=True,
                default_value='AVIA_QLOUD_OAUTH_TOKEN',
            )
            qloud_token_vault_owner = sdk2.parameters.String(
                'Vault secret owner',
                required=True,
                default_value='AVIA',
            )
            qloud_project_name = sdk2.parameters.String(
                'Qloud project name',
                required=True,
                default_value='avia',
            )

        with sdk2.parameters.Group('Stand parameters') as stand_params:
            source_qloud_environment_id = sdk2.parameters.String(
                'Source Qloud environment id',
                description='project.application.environment',
                required=True,
                default_value='avia.go-wizard.testing',
            )
            balancer_type = sdk2.parameters.String(
                'Qloud balancer type',
                description='testing-avia-wizard',
                required=True,
                default_value='testing-avia-wizard',
            )
            stands_application_name = sdk2.parameters.String(
                'Stands Qloud application name',
                required=True,
                default_value='wizard-stands',
            )
            stand_component_names = sdk2.parameters.List(
                'Stands Qloud application id',
                required=True,
                default=['app', 'organic'],
            )
            deploy_hook = sdk2.parameters.String(
                'Deploy hook',
                required=True,
                default_value='http://tools.production.avia.yandex.net/qloud/deploy/hook?type=test-stand',
            )
            package_config = sdk2.parameters.String(
                'Package config',
                description='project/service/pkg.json',
                required=True,
                default_value='travel/avia/wizard/stand_pkg.json',
            )
            resource_type = sdk2.parameters.String(
                'Resource type',
                required=True,
                default_value='AVIA_WIZARD_BINARY',
            )

        with sdk2.parameters.Group('Docker parameters') as docker_params:
            docker_token_vault_name = sdk2.parameters.String(
                'Docker token vault name',
                required=True,
                default_value='AVIA_DOCKER_REGISTRY_OAUTH_TOKEN',
            )
            docker_registry = sdk2.parameters.String(
                'Docker registry',
                required=True,
                default_value='registry.yandex.net',
            )
            docker_repository = sdk2.parameters.String(
                'Docker repository',
                required=True,
                default_value='avia',
            )
            docker_user = sdk2.parameters.String(
                'Docker user',
                required=True,
                default_value='robot-avia-api-pilot',
            )

    def on_execute(self):
        logging.info('Execution started')
        logging.info('Arcadia URL: %s', self.Parameters.arcadia_url)
        logging.info('Arcadia patch: %s', self.Parameters.arcadia_patch)
        logging.info('Arcanum review id: %s', self.Context.arcanum_review_id)

        qloud_token = sdk2.Vault.data(
            self.Parameters.qloud_token_vault_owner,
            self.Parameters.qloud_token_vault_name,
        )
        self.qloud_api = QloudPublicApi(qloud_token, timeout=QLOUD_API_TIMEOUT)
        self.qloud_private_api = QloudPrivateApi(qloud_token, timeout=QLOUD_API_TIMEOUT)

        if self._need_deploy_stand():
            logging.info('Start deploy stand')
            self._build_and_push_docker_image()
            try:
                self._deploy_stand()
            except Exception:
                msg = 'Error in deploy stand %s', self._stand_environment_id()
                logging.exception(msg)
                raise SandboxTaskFailureError(msg)
            logging.info('Done deploy stand')
        elif self._need_delete_stand():
            logging.info('Start delete stand')
            try:
                self._delete_stand()
            except Exception:
                msg = 'Error in delete stand %s', self._stand_environment_id()
                logging.exception(msg)
                raise SandboxTaskFailureError(msg)

            logging.info('Done delete stand')

        logging.info('Execution finished successfully')

    def _need_deploy_stand(self):
        # type: () -> bool
        return bool(self.Parameters.arcadia_patch)

    def _need_delete_stand(self):
        return 'TESTENV-COMMIT-CHECK' in self.Parameters.tags and self.qloud_api.environment_exists(self._stand_environment_id())

    def _delete_stand(self):
        self.qloud_api.delete_environment(self._stand_environment_id())

    def _arcadia_review_id(self):
        # type: () -> str
        if self.Context.arcanum_review_id:
            return str(self.Context.arcanum_review_id)
        review_merge_revision = svn.Arcadia.parse_url(self.Parameters.arcadia_url).revision
        logs = svn.Arcadia.log(self.Parameters.arcadia_url, revision_from=review_merge_revision, limit=1)
        revision_log_entry = logs[0]
        log_message = revision_log_entry['msg']
        return re.search(r'REVIEW: (\d*)', log_message).group(1)

    def _build_and_push_docker_image(self):
        with self.memoize_stage.create_children:
            self._run_ya_package()
        ya_package_task = self.find(id=self.Context.ya_package_task_id).first()
        if ya_package_task.status != ctt.Status.SUCCESS:
            raise SandboxTaskFailureError('Child YA_PACKAGE task %s failed', ya_package_task.id)
        logging.info('docker_image_version: %s', ya_package_task.Context.output_resource_version)
        self.Context.docker_image_version = ya_package_task.Context.output_resource_version

    def _run_ya_package(self):
        ya_package_task = sdk2.Task['YA_PACKAGE'](
            self,
            description='Build {} stand docker image. Child of task {}'.format(self.Parameters.application_title, self.id),
            notifications=self.Parameters.notifications,
            tags=self.Parameters.tags,
            create_sub_task=False,
            package_type='docker',
            docker_push_image=True,
            docker_registry=self.Parameters.docker_registry,
            docker_image_repository=self.Parameters.docker_repository,
            packages=self.Parameters.package_config,
            docker_user=self.Parameters.docker_user,
            docker_token_vault_name=self.Parameters.docker_token_vault_name,
            docker_build_network='host',
            resource_type=self.Parameters.resource_type,
            checkout=False,
            use_aapi_fuse=True,
            use_arc_instead_of_aapi=True,
            ya_yt_store=False,
            ignore_recurses=False,
            arcadia_patch=self.Parameters.arcadia_patch,
            custom_version=self._arcadia_review_id(),
            env_vars='DOCKER_BUILDKIT=1',
        ).enqueue()

        self.Context.ya_package_task_id = ya_package_task.id

        raise sdk2.WaitTask(
            [ya_package_task.id],
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=True,
            timeout=14400,
        )

    def _deploy_stand(self):
        stands_application_id = '{}.{}'.format(
            self.Parameters.qloud_project_name,
            self.Parameters.stands_application_name,
        )
        stand_name = self._stand_name()
        if not self.qloud_api.environment_exists(self._stand_environment_id()):
            self.qloud_api.create_environment(stands_application_id, stand_name)
        self.qloud_api.create_stand_domain(
            self.Parameters.stands_application_name,
            stand_name,
            self.Parameters.balancer_type
        )
        registry_image_name = self._get_registry_image_name()
        docker_hash = self.qloud_private_api.docker_hash(registry_image_name, self._arcadia_review_id())
        stand_environment_config = self._create_stand_environment_config(
            self.Parameters.source_qloud_environment_id,
            list(self.Parameters.stand_component_names),
            stand_name,
            self.Context.docker_image_version,
            docker_hash,
        )
        self.qloud_api.upload_environment(stand_environment_config)
        self.qloud_api.setup_deploy_hook(self._stand_environment_id(), self.Parameters.deploy_hook)

    def _get_registry_image_name(self):
        return self.Context.docker_image_version.split(':')[0].split('/', 1)[1]

    def _stand_name(self):
        return 'pr-{}'.format(self._arcadia_review_id())

    def _stand_environment_id(self):
        return '{}.{}.{}'.format(
            self.Parameters.qloud_project_name,
            self.Parameters.stands_application_name,
            self._stand_name()
        )

    def _create_stand_environment_config(
            self,
            environment_from_id,
            components,
            environment,
            docker_image,
            docker_hash,
    ):
        """
        :param environment_from_id: avia.go-wizard.testing
        :param application_to_id: avia.wizard-stands
        :param components: {app, wizard-organic}
        :param environment: pr-123
        :param docker_image: registry.yandex.net/avia/wizard-stands:123456
        :param docker_hash:
        :return:
        """
        config = self.qloud_api.dump_environment(environment_from_id)
        config['components'] = [c for c in config['components'] if c['componentName'] in components]
        config['objectId'] = self._stand_environment_id()
        config['comment'] = environment
        for c in config['components']:
            c['properties']['repository'] = docker_image
            c['properties']['hash'] = docker_hash
            c['instanceGroups'] = self._choose_instance_groups(c['instanceGroups'])

        return config

    @staticmethod
    def _choose_instance_groups(component_instance_groups):
        # type: (List[Dict[str, Any]]) -> List[Dict[str, Any]]

        for group in component_instance_groups:
            if group['units'] >= 1:
                new_group_config = deepcopy(group)
                new_group_config['units'] = 1
                return [new_group_config]

        return [
            {
                'backup': False,
                'location': u'VLADIMIR',
                'units': 1,
                'weight': 1
            }
        ]
