# -*- encoding: utf-8 -*-
import inspect
import logging
import shutil
from itertools import chain
from pathlib2 import Path

from sandbox import sdk2
from sandbox.sdk2.service_resources import SandboxTasksBinary
from sandbox.projects.common.ya_deploy.release_integration import ReleaseToNannyAndYaDeployTask2

from sandbox.projects.avia.base import AviaBaseTask


class AviaDumpData(ReleaseToNannyAndYaDeployTask2, AviaBaseTask):
    _resources = {}
    OUTPUT_PATH = './output'
    COMMON_DICTS_PATH = './output/common_dicts'
    COMMON_DICTS_ARCHIVE_NAME = 'common_dicts'

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Db parameters'):
            host = sdk2.parameters.String('Host name', required=True)
            user = sdk2.parameters.String('User', default='avia', required=True)
            database_name = sdk2.parameters.String('Database name', default='avia', required=True)
            password_vault = sdk2.parameters.String(
                'Password vault name', required=True, default_value='AVIA_MYSQL_DB_PASSWORD'
            )

        with sdk2.parameters.Group('Yt parameters'):
            yt_proxy = sdk2.parameters.String('Proxy', default='hahn', required=True)
            yt_token_vault = sdk2.parameters.String('Proxy', default='YT_TOKEN', required=True)

        vaults_owner = sdk2.parameters.String('Vaults owner', required=True, default_value='AVIA')
        build_common_dicts_archive = sdk2.parameters.Bool('Build common dicts archive', default=False)

    def on_save(self):
        self.Requirements.tasks_resource = (
            SandboxTasksBinary.find(attrs={'target': 'avia/sandbox/dump_data'}).first().id
        )

    def on_success(self, prev_status):
        super(AviaDumpData, self).on_success(prev_status)
        self.on_release(
            dict(
                releaser=self.author,
                release_status='stable',
                release_subject='Auto release AviaDumpData task',
                email_notifications={'to': [], 'cc': []},
                release_comments='Auto release AviaDumpData task',
            )
        )

    def on_execute(self):
        logging.info('Start task')
        self._resources = self._load_resources()

        output_dir = self._get_output_dir()
        self._import_data(output_dir)

        output_files = [p for p in output_dir.iterdir() if p.is_file()]
        for output_file in output_files:
            self._upload_resource(output_file)

        if self.Parameters.build_common_dicts_archive:
            self._upload_common_dicts_archive(output_files)

        logging.info('Done')

    def _import_data(self, output_dir):
        from travel.avia.dump_data.lib import new_dumper

        db_password = sdk2.Vault.data(self.Parameters.vaults_owner, self.Parameters.password_vault)
        yt_token = sdk2.Vault.data(self.Parameters.vaults_owner, self.Parameters.yt_token_vault)
        dumper = new_dumper(
            host=self.Parameters.host,
            password=db_password,
            user=self.Parameters.user,
            database=self.Parameters.database_name,
            yt_proxy=self.Parameters.yt_proxy,
            yt_token=yt_token,
            output_dir=output_dir,
        )
        dumper.dump_all()

    def _upload_resource(self, output_file):
        output_file_absolute_path = output_file.absolute()
        logging.info('Uploading %s', output_file_absolute_path)

        recource_name = resource_name(output_file.name)
        resource = self._resources.get(recource_name)(
            self,
            description=recource_name,
            path=recource_name + '.data',
            ttl=30,
        )
        if not resource:
            raise Exception('Can\'t find resource {!r}'.format(recource_name))

        resource_data = sdk2.ResourceData(resource)
        shutil.copy(
            src=output_file_absolute_path.as_posix(),
            dst=resource_data.path.absolute().as_posix(),
        )
        resource_data.ready()

    def _upload_common_dicts_archive(self, output_files):
        from sandbox.projects.Travel.resources.dicts import TRAVEL_AVIA_COMMON_DICTS_BUNDLE

        common_dicts_dir = Path(self.COMMON_DICTS_PATH)
        common_dicts_dir.mkdir(parents=True, exist_ok=True)
        logging.info('Copying common dicts into %s', common_dicts_dir.absolute().as_posix())

        for output_file in output_files:
            try:
                output_file_absolute_path = output_file.absolute()
                base_name = output_file.name
                resource_class = self._resources.get(resource_name(base_name))
                if resource_class is None:
                    logging.error("Can't find resource_class for resource name %s", base_name)
                    continue

                if resource_class.is_common_dict:
                    logging.info('Copying %s', output_file_absolute_path.as_posix())
                    new_filename = output_file_absolute_path.as_posix().replace('.bin', '.data')
                    shutil.copy(
                        src=output_file_absolute_path.as_posix(),
                        dst=new_filename,
                    )
                    shutil.move(
                        src=new_filename,
                        dst=common_dicts_dir.absolute().as_posix(),
                    )
            except Exception:
                logging.exception('Error while adding %s to common dicts directory', output_file)
                raise

        try:
            logging.info('Building archive')
            archive_path = shutil.make_archive(
                self.COMMON_DICTS_ARCHIVE_NAME, 'gztar', root_dir=self.COMMON_DICTS_PATH, base_dir='.'
            )
            logging.info('Archive has been built: %s', archive_path)
            archive_path = Path(archive_path)
            base_name = archive_path.absolute().name
            resource = TRAVEL_AVIA_COMMON_DICTS_BUNDLE(self, description=base_name, path=base_name, ttl='inf')
            resource_data = sdk2.ResourceData(resource)
            logging.info('Created common dicts resource data with path: %s', resource_data.path.absolute().as_posix())
            resource_data.ready()
        except Exception:
            logging.exception('Error while uploading common dicts archive')
            raise

    @staticmethod
    def _load_resources():
        from sandbox.projects.Travel.resources import dicts
        from sandbox.projects.avia_resources import resources

        result = {}
        for name, entry in chain(dicts.__dict__.items(), resources.__dict__.items()):
            if (
                inspect.isclass(entry)
                and issubclass(entry, dicts.TRAVEL_DICT_AVIA_BASE)
                and entry != dicts.TRAVEL_DICT_AVIA_BASE
                and hasattr(entry, 'resource_name')
            ):
                result[entry.resource_name] = entry
        return result

    def _get_output_dir(self):
        output_dir = Path(self.OUTPUT_PATH)
        output_dir.mkdir(parents=True, exist_ok=True)
        logging.info('Created directory {}'.format(output_dir.absolute()))
        return output_dir


def resource_name(file_name):
    name, _ = file_name.split('.')
    return name
