# -*- encoding: utf-8 -*-
import gzip
import logging
import os
import shutil
import tarfile

from pathlib2 import Path

from sandbox import sdk2
from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia_resources.resources import AVIA_MYSQL_SCHEMA_DUMP, AVIA_MYSQL_PACKAGE
from sandbox.projects.common import binary_task


class AviaDumpMysql(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    DUMP_FILE = 'schema_dump.sql'
    ARCHIVE_FILE = 'schema_dump.gz'
    MYSQL_DIR = 'mysql-5.7.29'

    class Parameters(sdk2.Task.Parameters):
        binary_release = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Db parameters'):
            mysql_package = sdk2.parameters.Resource(
                "Archive with mysql binaries",
                resource_type=AVIA_MYSQL_PACKAGE
            )
            mysql_host = sdk2.parameters.String('Mysql host name', required=True)
            mysql_port = sdk2.parameters.String('Mysql port', required=True, default=3306)
            mysql_user = sdk2.parameters.String('Mysql user', default='avia', required=True)
            database_name = sdk2.parameters.String('Database name', default='avia', required=True)
            password_vault = sdk2.parameters.YavSecret('Mysql password')

    class Context(sdk2.Context):
        mysqldump = None

    @property
    def mysqldump(self):
        if self.Context.mysqldump is None:
            if not self.Parameters.mysql_package:
                self.Context.mysqldump = 'mysqldump'

            else:
                mysql_package = str(sdk2.ResourceData(self.Parameters.mysql_package).path)
                mysql_path = str(self.path('mysql_pack'))
                with tarfile.open(mysql_package, 'r:gz') as tf:
                    logging.info('Extractinging mysql to %s', mysql_path)
                    tf.extractall(mysql_path)
                self.Context.mysqldump = str(self.path('mysql_pack', self.MYSQL_DIR, 'bin', 'mysqldump'))

        logging.info('Using mysqldump executable: %s', self.Context.mysqldump)
        return self.Context.mysqldump

    def on_execute(self):
        from travel.avia.library.python.mysqlutils import dump_db

        logging.info('Start task')

        dump_db(
            self.get_conection_params(), self.DUMP_FILE,
            mysqldump_executable=self.mysqldump,
        )
        logging.info('Dump size: %s', os.path.getsize(self.DUMP_FILE))

        self.upload_resource()
        logging.info('Done')

    def get_conection_params(self):
        return {
            'host': self.Parameters.mysql_host,
            'port': self.Parameters.mysql_port,
            'user': self.Parameters.mysql_user,
            'passwd': self.Parameters.password_vault.data()['password'],
            'db': self.Parameters.database_name,
        }

    def upload_resource(self):
        output_file = Path(self.DUMP_FILE)
        archive_file = Path(self.ARCHIVE_FILE)
        assert output_file.is_file()

        output_file_absolute_path = output_file.absolute()
        logging.info('Uploading %s', output_file_absolute_path)

        resource = AVIA_MYSQL_SCHEMA_DUMP(
            self,
            description=archive_file.name,
            path=archive_file.name,
            ttl='inf',
        )

        with open(str(output_file.absolute()), 'rb') as f_in:
            with gzip.open(str(archive_file.absolute()), 'wb') as f_out:
                shutil.copyfileobj(f_in, f_out)

        resource_data = sdk2.ResourceData(resource)
        resource_data.ready()
