# -*- encoding: utf-8 -*-
import datetime
import logging
import zlib

from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.projects.common import binary_task
from sandbox.projects.common.ya_deploy.release_integration import ReleaseToNannyAndYaDeployTask2

from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib.yt_helpers import YtClientFactory


class AviaDumpPartnerConversions(ReleaseToNannyAndYaDeployTask2, binary_task.LastBinaryTaskRelease, AviaBaseTask):
    class Requirements(sdk2.Requirements):
        # configure this for your task, the more accurate - the better
        cores = 1  # exactly 1 core
        disk_space = 128  # 128 Megs or less
        ram = 1024  # 1024 Megs or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        # binary task release parameters
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Yt parameters'):
            yt_proxy = sdk2.parameters.String('Proxy', default='hahn', required=True)
            yt_token = sdk2.parameters.YavSecret(
                'Yav secret with YT token',
                required=True,
                default='sec-01dfxmszhq27tk66hm107d0ycd',
            )

        yp_token_yav_secret_id = sdk2.parameters.String(
            'Yav secret with YP token',
            required=True,
            default='sec-01fevjyb9eq5jxp249wc6pmgbe',
        )
        yt_reference_path = sdk2.parameters.String(
            'YT reference path',
            required=True,
            default_value='//home/avia/reports/conversion_partners_sources/',
        )

    def _get_path(self):
        return self.Parameters.yt_reference_path + datetime.date.today().isoformat()

    def on_execute(self):
        logging.info('Start task')

        self._dump()

        logging.info('Done')

    def _dump(self):
        yt_client = YtClientFactory.create(self.Parameters.yt_proxy, self.Parameters.yt_token.data()['token'])
        conversions = self._read(yt_client, self._get_path())
        compressed_data = self._compress(conversions)
        self._create_resource(compressed_data)
        self._release()

    def _read(self, yt_client, path):
        import yt.wrapper as yt
        from travel.proto.dicts.avia.conversion_pb2 import TConversion, TConversions
        conversions = TConversions()
        for i, line in enumerate(yt_client.read_table(yt.TablePath(path, columns=('source', 'conversion', 'billing_order_id')))):
            conversion = TConversion()
            conversion.BillingOrderID = line['billing_order_id']
            conversion.Source = line['source']
            conversion.Conversion = float(line['conversion'])
            conversions.Conversions.append(conversion)
        return conversions

    @staticmethod
    def _compress(conversions):
        raw_data = conversions.SerializeToString()
        return zlib.compress(raw_data)

    def _create_resource(self, compressed_data):
        from sandbox.projects.avia_resources.resources import AVIA_PARTNER_CONVERSIONS
        resource_data = sdk2.ResourceData(AVIA_PARTNER_CONVERSIONS(
            self,
            description='Partner conversions',
            path='partner_conversions.data',
            ttl=30,
        ))
        resource_data.path.write_bytes(compressed_data)
        resource_data.ready()

    def _release(self):
        self.on_release(dict(
            releaser=self.author,
            release_status=ctt.ReleaseStatus.STABLE,
            email_notifications=dict(to=[], cc=[]),
            release_comments='Partner conversions release',
        ))
