# coding=utf-8
from __future__ import unicode_literals

import logging
import json
import shutil
from pathlib2 import Path

from lxml import etree
from sandbox import sdk2
from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.common import binary_task
from sandbox.projects.Travel.resources.dicts import TRAVEL_DICT_AVIA_FARE_FAMILIES_PROD
from sandbox.sandboxsdk.svn import Arcadia


logger = logging.getLogger(__name__)


class AviaPrepareFareFamilyResourceTask(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    OUTPUT_PATH = './output'

    class Requirements(sdk2.Requirements):
        # configure this for your task, the more accurate - the better
        cores = 1  # exactly 1 core
        disk_space = 128  # 128 Megs or less
        ram = 128  # 128 Megs or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):

        # binary task release parameters
        ext_params = binary_task.binary_release_parameters(stable=True)
        configs_path = sdk2.parameters.SvnUrl(
            'Path to fare_families configs in arcadia',
            required=True,
            default_value=Arcadia.trunk_url('travel/avia/ticket_daemon_api/data/fare_families')
        )
        external_xpath_configs_path = sdk2.parameters.SvnUrl(
            'Path to fare_families_expressions configs in arcadia',
            required=True,
            default_value=Arcadia.trunk_url('travel/avia/ticket_daemon_api/data/fare_families_expressions')
        )
        resource_ttl = sdk2.parameters.Integer('Resource TTL', required=True, default=30)

    def on_execute(self):
        """A simple task which writes TFareFamilies to a file"""
        from travel.library.python.dicts import file_util

        super(AviaPrepareFareFamilyResourceTask, self).on_execute()

        converter = JsonToPbConverter()

        output_dir = self._get_output_dir()
        file_name = output_dir / TRAVEL_DICT_AVIA_FARE_FAMILIES_PROD.resource_name

        with open(str(file_name), 'wb') as file:
            for airline_ff_file_name, content in self._build_config():
                airline_id, _ = airline_ff_file_name.split('_', 1)
                fare_families_proto = converter.convert(content, int(airline_id))
                file_util.write_binary_string(file, fare_families_proto.SerializeToString())

        self._upload_resource(file_name)

    def _build_config(self):
        configs = Arcadia.list(self.Parameters.configs_path, as_list=True)
        ext_configs = Arcadia.list(self.Parameters.external_xpath_configs_path, as_list=True)
        logger.info('Prepare configs for dump (%r)', configs)
        for config_file_name in configs:
            ext_xpath_filename = config_file_name.replace('.json', '.xml')
            ext_xpath_expressions = {}
            if ext_xpath_filename in ext_configs:
                ext_config = Arcadia.cat(self.Parameters.external_xpath_configs_path + '/' + ext_xpath_filename)
                ext_xpath_expressions = self._get_xpath_expressions(ext_config)

            config = json.loads(Arcadia.cat(self.Parameters.configs_path + '/' + config_file_name))
            self._fill_rules(config, ext_xpath_expressions)
            yield config_file_name.rstrip('.json'), config

    @staticmethod
    def _get_xpath_expressions(ext_config):
        expressions = {}
        parser = etree.XMLParser(remove_comments=True)
        tree = etree.fromstring(ext_config, parser=parser)
        for expression in tree:
            expressions[expression.get('id')] = expression.text

        return expressions

    @staticmethod
    def _fill_rules(fare_families, ext_xpaths):
        """Replace external_xpath_ref with related xpath and remove ignored rules"""
        for ff in fare_families:
            for term in ff['terms']:
                rules = []

                for rule in term['rules']:
                    if rule.get('ignore', False) is True:
                        continue
                    if 'external_xpath_ref' in rule:
                        # Считаем, что всегда есть, иначе тесты бы не прошли
                        rule['xpath'] = ext_xpaths[rule['external_xpath_ref']]
                        rule.pop('external_xpath_ref')
                    rules.append(rule)
                term['rules'] = rules

    def _get_output_dir(self):
        output_dir = Path(self.OUTPUT_PATH)
        output_dir.mkdir(parents=True, exist_ok=True)
        logger.info('Created directory {}'.format(output_dir.absolute()))
        return output_dir

    def _upload_resource(self, output_file):
        output_file_absolute_path = output_file.absolute()
        logger.info('Uploading %s', output_file_absolute_path)

        resource_name = output_file.name
        resource = TRAVEL_DICT_AVIA_FARE_FAMILIES_PROD(
            self,
            description=resource_name,
            path=resource_name + '.data',
            ttl=self.Parameters.resource_ttl,
        )
        if not resource:
            raise Exception('Can\'t find resource {!r}'.format(resource_name))

        resource_data = sdk2.ResourceData(resource)
        shutil.copy(
            src=output_file_absolute_path.as_posix(),
            dst=resource_data.path.absolute().as_posix(),
        )
        resource_data.ready()


class JsonToPbConverter(object):
    def convert(self, airline_fare_families, airline_id):
        """Convert from JSON to protobuf"""
        from travel.proto.avia.fare_families.fare_families_pb2 import TFareFamilies, TLocalizedText, TFareFamily

        result_fare_families = TFareFamilies()
        result_fare_families.AirlineId = int(airline_id)
        for fare_family in airline_fare_families:
            result_fare_family = TFareFamily()
            result_fare_family.BaseClass = fare_family['base_class']
            result_fare_family.TariffCodePattern = fare_family['tariff_code_pattern']
            result_fare_family.TariffGroupName.CopyFrom(
                TLocalizedText(Values=fare_family['tariff_group_name'])
            )
            result_fare_family.Brand = fare_family['brand']
            result_fare_family.Terms.extend(list(self._convert_terms(fare_family['terms'])))

            result_fare_families.Values.extend([result_fare_family])
        return result_fare_families

    def _convert_terms(self, terms):
        from travel.proto.avia.fare_families.fare_families_pb2 import TTerm

        for term in terms:
            result_term = TTerm()
            result_term.Code = term['code']
            result_term.SpecialNotes.extend(list(self._convert_special_notes(term.get('special_notes', []))))
            result_term.Rules.extend(list(self._convert_rules(term.get('rules', []))))
            yield result_term

    def _convert_rules(self, rules):
        from travel.proto.avia.fare_families.fare_families_pb2 import TRule

        for rule in rules:
            result = TRule()

            result.Availability = rule.get('availability', '')
            result.Comment = rule.get('comment', '')
            result.Xpath = rule.get('xpath', '')
            if 'charge' in rule:
                result.Charge.CopyFrom(self._convert_price(rule['charge']))
            result.SpecialNotes.extend(list(self._convert_special_notes(rule.get('special_notes', []))))
            result.Places = rule.get('places', 0)
            result.Weight = rule.get('weight', 0)
            result.Size = rule.get('size', '')

            yield result

    def _convert_price(self, price):
        from travel.proto.commons_pb2 import TPrice, ECurrency

        result_price = TPrice()
        value = price['value']
        e_currency = 'C_{}'.format(price['currency'])
        result_price.Currency = ECurrency.Value(e_currency)
        result_price.Amount = int(''.join(value.split('.')))
        result_price.Precision = len(value.split('.')[1]) if len(value.split('.')) == 2 else 0
        return result_price

    def _convert_special_notes(self, special_notes):
        from travel.proto.avia.fare_families.fare_families_pb2 import TLocalizedText

        for special_note in special_notes:
            yield TLocalizedText(Values=special_note)
