# -*- encoding: utf-8 -*-
from datetime import datetime, timedelta

import sandbox.common.types.misc as ctm
from sandbox.projects.common import binary_task
from sandbox import sdk2

from sandbox.projects.avia.base import AviaBaseTask


class AviaFetchFlightStatsCreatedAlertsResource(sdk2.Resource):
    auto_backup = True
    any_arch = True
    alerts_date = sdk2.Attributes.String('alerts_date', required=True)
    environment = sdk2.Attributes.String('environment', required=True)


class AviaFlightStatsRegistrar(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass  # We do not need caches

        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        DEPARTURE_DAY_KEY = 'departure_day'
        TOMORROW_KEY = 'tomorrow'

        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Common settings') as common_settings:
            yt_proxy = sdk2.parameters.String(
                'Reference YT proxy',
                required=True,
                default_value='hahn',
            )
            yt_token = sdk2.parameters.YavSecret(
                'Yav secret with Yt token',
                required=True,
            )
            avia_backend_url = sdk2.parameters.String(
                'Avia-backend url',
                required=True,
                default_value='http://backend.testing.avia.yandex.net',
            )
            shared_flights_api_url = sdk2.parameters.String(
                'Shared flights api base url',
                required=True,
                default_value='http://shared-flights.testing.avia.yandex.net/api/v1',
            )
            vaults_owner = sdk2.parameters.String('Vaults owner', required=True, default_value='AVIA')

        with sdk2.parameters.Group('Fetch settings') as fetching_settings:
            popular_flights_yt_table = sdk2.parameters.String(
                'Popular flights YT table',
                required=True,
                default_value='//home/avia/reports/wizard/popular_flights_with_competitors',
            )
            flights_limit = sdk2.parameters.Integer(
                'Flights\' limit to register in one run',
                required=True,
                default_value=100,
            )
            with sdk2.parameters.RadioGroup('Fetch type') as fetch_type:
                fetch_type.values[DEPARTURE_DAY_KEY] = fetch_type.Value('Departure day', default=True)
                fetch_type.values[TOMORROW_KEY] = fetch_type.Value('Fetch for tomorrow')

                with fetch_type.value[DEPARTURE_DAY_KEY]:
                    departure_day = sdk2.parameters.StrictString(
                        'Departure day YYYY-MM-DD',
                        regexp="\d{4}-\d{2}-\d{2}",
                        required=True,
                    )
            with sdk2.parameters.RadioGroup('Environment') as environment:
                environment.values['testing'] = environment.Value(value='testing', default=True)
                environment.values['production'] = 'production'

        with sdk2.parameters.Group('FlightStats settings') as flight_stats_settings:
            flight_stats_create_alert = sdk2.parameters.Bool('Flight stats create alert', default_value=False)
            flight_stats_app_id = sdk2.parameters.String('Flight stats app ID', required=True, default_value='32c60eaa')
            flight_stats_app_key_vault = sdk2.parameters.String(
                'Flight stats app key vault',
                required=True,
                default_value='AVIA_FLIGHT_STATS_APP_KEY'
            )
            flight_stats_alert_handler = sdk2.parameters.String(
                'Flight stats alert handler',
                required=False,
                default='',
                default_value='https://avia.yandex.ru/flight-stats/alert'
            )

            http_connection_timeout = sdk2.parameters.Integer(
                'Http connection timeout in seconds',
                required=False,
                default_value=60,
            )
            http_read_timeout = sdk2.parameters.Integer(
                'Http read timeout in seconds',
                required=False,
                default_value=60,
            )
            flight_stats_url = sdk2.parameters.String(
                'FlightStats url',
                required=True,
                default_value='https://api.flightstats.com/flex'
            )

    def on_execute(self):
        from travel.avia.flight_status_registrar.flight_stats.lib.pipeline import flight_stats_pipeline

        super(AviaFlightStatsRegistrar, self).on_execute()

        if self.Parameters.fetch_type == self.Parameters.TOMORROW_KEY:
            departure_date = (datetime.now() + timedelta(days=1)).date()
        else:
            departure_date = datetime.strptime(str(self.Parameters.departure_day), '%Y-%m-%d').date()

        flight_stats_app_key = sdk2.Vault.data(self.Parameters.vaults_owner, self.Parameters.flight_stats_app_key_vault)
        yt_token = self.Parameters.yt_token.data()['token']
        flight_stats_pipeline(
            api_url=self.Parameters.flight_stats_url,
            app_id=self.Parameters.flight_stats_app_id,
            app_key=flight_stats_app_key,
            yt_proxy=self.Parameters.yt_proxy,
            yt_token=yt_token,
            popular_flights_table=self.Parameters.popular_flights_yt_table,
            flights_per_request=100,
            avia_backend_url=self.Parameters.avia_backend_url,
            shared_flights_api_url=self.Parameters.shared_flights_api_url,
            departure_date=departure_date,
            flights_limit=self.Parameters.flights_limit,
            http_connection_timeout=self.Parameters.http_connection_timeout,
            http_read_timeout=self.Parameters.http_read_timeout,
            environment=self.Parameters.environment,
            alert_handler=self.Parameters.flight_stats_alert_handler,
            create_alert=self.Parameters.flight_stats_create_alert,
            task=self,
            created_alerts_resource_type=AviaFetchFlightStatsCreatedAlertsResource,
        )
