# -*- encoding: utf-8 -*-
from datetime import datetime, timedelta

import sandbox.common.types.misc as ctm
from sandbox.projects.common import binary_task
from sandbox import sdk2

from sandbox.projects.avia.base import AviaBaseTask


class AviaOagFlightsRegistrar(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass  # We do not need caches

        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        DEPARTURE_DAY_KEY = 'departure_day'
        TOMORROW_KEY = 'tomorrow'

        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Common settings') as common_settings:
            reference_yt_proxy = sdk2.parameters.String(
                'Reference YT proxy',
                required=True,
                default_value='hahn'
            )
            yt_token = sdk2.parameters.YavSecret(
                'Yav secret with Yt token',
                required=True
            )
            avia_backend_url = sdk2.parameters.String(
                'Avia-backend url',
                required=True,
                default_value='http://backend.testing.avia.yandex.net'
            )
            shared_flights_url = sdk2.parameters.String(
                'Shared flights url',
                required=True,
                default_value='http://shared-flights.testing.avia.yandex.net'
            )

        with sdk2.parameters.Group('Fetch settings') as fetching_settings:
            popular_flights_yt_table = sdk2.parameters.String(
                'Popular flights YT table',
                required=True,
                default_value='//home/avia/reports/wizard/popular_flights_with_competitors'
            )
            flights_limit = sdk2.parameters.Integer(
                'Flights\' limit to register in one run',
                required=True,
                default_value=10
            )
            with sdk2.parameters.RadioGroup('Fetch type') as fetch_type:
                fetch_type.values[DEPARTURE_DAY_KEY] = fetch_type.Value('Departure day', default=True)
                fetch_type.values[TOMORROW_KEY] = fetch_type.Value('Fetch for tomorrow')

                with fetch_type.value[DEPARTURE_DAY_KEY]:
                    departure_day = sdk2.parameters.StrictString(
                        'Departure day YYYY-MM-DD',
                        regexp="\d{4}-\d{2}-\d{2}",
                        required=True
                    )

            oag_credentials = sdk2.parameters.YavSecret(
                'Yav secret with OAG credentials',
                default='sec-01e5s8mb0y8kwy3vxz77w08v36'
            )

    def on_execute(self):
        from travel.avia.flight_status_registrar.oag.lib.pipeline import oag_pipeline

        super(AviaOagFlightsRegistrar, self).on_execute()

        if self.Parameters.fetch_type == self.Parameters.TOMORROW_KEY:
            departure_date = (datetime.now() + timedelta(days=1)).date()
        else:
            departure_date = datetime.strptime(str(self.Parameters.departure_day), '%Y-%m-%d').date()

        oag_credentials = self.Parameters.oag_credentials.data()
        yt_token = self.Parameters.yt_token.data()['token']

        oag_pipeline(
            app_id=oag_credentials['app_id'],
            auth_key=oag_credentials['app_key'],

            yt_proxy=self.Parameters.reference_yt_proxy,
            yt_token=yt_token,
            popular_flights_table=self.Parameters.popular_flights_yt_table,

            shared_flights_url=self.Parameters.shared_flights_url,
            flights_per_request=100,
            avia_backend_url=self.Parameters.avia_backend_url,

            departure_date=departure_date,
            flights_limit=self.Parameters.flights_limit,
        )
