import logging
import os
import shutil
import subprocess

from sandbox import sdk2
from sandbox.common.errors import SandboxEnvironmentError
from sandbox.sdk2.vcs.git import Git
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.avia.base import AviaBaseTask


class GenerateAviaSitemaps(AviaBaseTask):
    GIT_URLS = [
        ('https://github.yandex-team.ru/avia/avia_data.git', 'avia_data', 'master', '609a9527f29098619de8267ea34eeaa0691603dc'),
        ('https://github.yandex-team.ru/avia/common.git', 'common', 'master', '6099a6927b513fbd3f395c5c23a6d992506a1e43'),
    ]

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Vault owner') as vault_block:
            vault_owner = sdk2.parameters.String('Vaults owner', required=True)

        with sdk2.parameters.Group('MYSQl settings') as mysql_block:
            db_user = sdk2.parameters.String('MYSQL database user', required=True)
            db_host = sdk2.parameters.String('MYSQL host', required=False, default=None)
            conductor_group = sdk2.parameters.String('Conductor group', required=False, default=None)
            db_password_vault_name = sdk2.parameters.String('MYSQL password vault name', required=True)

        with sdk2.parameters.Group('MDS settings') as mds_block:
            mds_endpoint = sdk2.parameters.String('MDS endpoint url', required=True)
            mds_bucket = sdk2.parameters.String('MDS bucket name', required=True)
            aws_access_key_vault_name = sdk2.parameters.String('AWS access secret key vault name', required=True)
            aws_access_key_id_vault_name = sdk2.parameters.String('AWS key id vault name', required=True)

    def on_execute(self):
        self._validate_params()
        for repository, checkout_path, branch, commit in self.GIT_URLS:
            Git(repository).clone(checkout_path, branch=branch, commit=commit)

        self._prepare_env()

        work_dir = str(self.path())
        shutil.copy(os.path.join(os.path.dirname(__file__), 'settings.py'), os.path.join(work_dir, 'settings.py'))

        my_env = os.environ.copy()
        my_env.update({
            'PYTHONPATH': work_dir + os.pathsep + my_env['PYTHONPATH'],
            'AVIA_TEMPLATE_DIR': os.path.dirname(__file__),
            'AVIA_DB_USER': self.Parameters.db_user,
            'AVIA_DB_PASSWORD': sdk2.Vault.data(self.Parameters.vault_owner, self.Parameters.db_password_vault_name),
            'DJANGO_SETTINGS_MODULE': 'settings',
            'AVIA_MEDIA_ROOT': work_dir,
            'MDS_S3_ENDPOINT_URL': self.Parameters.mds_endpoint,
            'MDS_S3_AVIA_ADMIN_BUCKET': self.Parameters.mds_bucket,
            'AWS_ACCESS_KEY_ID': sdk2.Vault.data(self.Parameters.vault_owner, self.Parameters.aws_access_key_id_vault_name),
            'AWS_SECRET_ACCESS_KEY': sdk2.Vault.data(self.Parameters.vault_owner, self.Parameters.aws_access_key_vault_name),
            'LOG_PATH': work_dir,
        })

        if self.Parameters.db_host:
            logging.info('Use mysql single host: %s', self.Parameters.db_host)
            my_env['AVIA_DB_HOST'] = self.Parameters.db_host

        else:
            logging.info('Use myql group: %s', self.Parameters.conductor_group)
            my_env['AVIA_DB_GROUP'] = self.Parameters.conductor_group

        child = run_process([
            self.venv.executable,
            os.path.join(os.path.dirname(__file__), 'gen_sitemap.py')
        ], stderr=subprocess.PIPE, environment=my_env, log_prefix='script_output')
        (stdout, stderr) = child.communicate()
        logging.info('Process stdout: %s', stdout)
        logging.info('Process stderr: %s', stderr)
        if child.returncode != 0:
            raise SandboxTaskFailureError('Script return code is {}'.format(child.returncode))

    def _validate_params(self):
        if self.Parameters.db_host is None and self.Parameters.conductor_group is None:
            raise ValueError('DB host or Conductor group should be set')

    def _setup_from_file(self, inp, venv):
        """
        Install dependencies from given file to given virtualenv.
        Support packages without version and packages with precies version ('==').
        """
        for line in inp:
            parts = line.split('==')
            version = None
            if len(parts) > 1:
                version = parts[1]

            try:
                environments.PipEnvironment(parts[0], version=version, venv=venv, use_wheel=True).prepare()
            except SandboxEnvironmentError:
                environments.PipEnvironment(parts[0], version=version, venv=venv, use_wheel=False).prepare()

    def _prepare_env(self):
        with environments.VirtualEnvironment(do_not_remove=True) as venv:
            for _, checkout_path, _, _ in self.GIT_URLS:
                path = os.path.join(checkout_path, 'requirements.txt')
                logging.info('Requirements from %s', path)
                with open(path) as inp:
                    self._setup_from_file(inp, venv)
            with open(os.path.join(os.path.dirname(__file__), 'requirements.txt'), 'rb') as inp:
                self._setup_from_file(inp, venv)
            self.venv = venv
