# -*- encoding: utf-8 -*-
from __future__ import unicode_literals

import logging

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.common.types import resource as ctr
from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.Travel.resources import dicts

log = logging.getLogger(__name__)


class AviaGenerateTravelSitemap(binary_task.LastBinaryTaskRelease, AviaBaseTask):
    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        disk_space = 128  # 128 Megs or less
        ram = 128  # 128 Megs or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):

        # binary task release parameters
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Yt parameters'):
            yt_proxy = sdk2.parameters.String('Proxy', default='hahn', required=True)
            yt_token = sdk2.parameters.YavSecret('YT token', default='sec-01dfxmszhq27tk66hm107d0ycd', required=True)

        with sdk2.parameters.Group('Sitemap parameters'):
            portal_url = sdk2.parameters.String(
                'Portal url',
                default='https://travel-test.yandex.ru',
                required=True,
            )
            changefreq = sdk2.parameters.String('changefreq text')
            add_lastmod = sdk2.parameters.Bool('Add last modification date', default=False)
            need_generate_flight_sitemap = sdk2.parameters.Bool('Need generate flight sitemap', default=False)
            need_add_avia_root = sdk2.parameters.Bool('Need add travel/avia root', default=False)
            need_generate_city_to_sitemap = sdk2.parameters.Bool('Need generate city to sitemap', default=False)

        with sdk2.parameters.Group('S3 paramters'):
            s3_endpoint = sdk2.parameters.String('S3 endpoint', default='https://s3.mdst.yandex.net', required=True)
            s3_prefix = sdk2.parameters.String('S3 prefix', default='sitemaps', required=True)
            s3_bucket = sdk2.parameters.String('S3 bucket', default='avia-indexer', required=True)
            s3_credentials = sdk2.parameters.YavSecret(
                'YT token',
                default='sec-01dsey00pc7fs9mdjxgt8nn3xy',
                required=True,
            )

    def on_execute(self):
        super(AviaGenerateTravelSitemap, self).on_execute()
        from travel.avia.library.python.sitemap_generator.lib.generator import SitemapGenerator
        from travel.library.python.dicts.settlement_repository import SettlementRepository
        from travel.library.python.dicts.avia.top_flight_repository import TopFlightRepository

        settlement_repository = SettlementRepository()
        settlement_resource_path = self._get_latest_resource_version_path(dicts.TRAVEL_DICT_RASP_SETTLEMENT_PROD)
        settlement_repository.load_from_file(settlement_resource_path)

        top_flight_repository = TopFlightRepository()
        top_flight_resource_path = self._get_latest_resource_version_path(dicts.TRAVEL_DICT_AVIA_TOP_FLIGHT_PROD)
        top_flight_repository.load_from_file(top_flight_resource_path)

        s3_credentials = self.Parameters.s3_credentials.data()
        s3_access_key = s3_credentials['aws_access_key_id']
        s3_secret_access_key = s3_credentials['aws_secret_access_key']

        yt_token = self.Parameters.yt_token.data()['token']

        generator = SitemapGenerator(
            portal_url=self.Parameters.portal_url,
            yt_proxy=self.Parameters.yt_proxy,
            yt_token=yt_token,
            changefreq=self.Parameters.changefreq,
            add_lastmod=self.Parameters.add_lastmod,
            s3_endpoint=self.Parameters.s3_endpoint,
            s3_prefix=self.Parameters.s3_prefix,
            s3_access_key=s3_access_key,
            s3_access_secret_key=s3_secret_access_key,
            s3_bucket=self.Parameters.s3_bucket,
            settlement_repository=settlement_repository,
            top_flight_repository=top_flight_repository,
            need_generate_flight_sitemap=self.Parameters.need_generate_flight_sitemap,
            need_add_avia_root=self.Parameters.need_add_avia_root,
            need_generate_city_to_sitemap=self.Parameters.need_generate_city_to_sitemap,
        )

        generator.generate()

    @staticmethod
    def _get_latest_resource_version_path(resource_class):
        resource = resource_class.find(state=ctr.State.READY).first()
        resource_data = sdk2.ResourceData(resource)
        return resource_data.path.as_posix()
