# coding=utf-8
import base64
import cStringIO
import csv
import logging
import time
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib.logs import configure_logging, get_sentry_dsn
from sandbox.projects.avia.lib.yt_helpers import YtClientFactory


# Change with caution! Stability in other version is not guranteed
BROWSER_CAPABILITIES = {
    'platform': 'ANY', 'browserName': 'chrome', 'version': '70'
}
SELENIUM_GRID_URL = 'http://selenium:selenium@sg.yandex-team.ru:4444/wd/hub'


INT_COLUMNS = {
    'CLICKS',
    'GROSS CLICKS',
    'UNIQUE CLICKS',
}


DOUBLE_COLUMNS = {
    'IMPRESSIONS',
    'PAYOUT',
    'CONVERSIONS',
    'CTR',
    'CR',
    'CPC',
    'CPA',
}


def _get_type(field):
    if field in INT_COLUMNS:
        return 'int64'

    if field in DOUBLE_COLUMNS:
        return 'double'

    return 'string'


def generate_yt_schema_by_fields(fields):
    return [
        {'name': field, 'type': _get_type(field)}
        for field in fields
    ]


def get_file_names_chrome(driver):
    if not driver.current_url.startswith("chrome://downloads"):
        driver.get("chrome://downloads/")

    s = driver.execute_script("""
        items = downloads.Manager.get().items_
        return items[0]
    """)

    if s['state'] == 'COMPLETE':
        return s.get('fileUrl') or s.get('file_url')  # Depends on Chrome version

    return None


class DownloadFail(Exception):
    def __init__(self, msg, status, console_log):
        super(DownloadFail, self).__init__(msg)
        self.status = status
        self.console_log = console_log


def get_file_content_chrome(driver, uri):
    """
        Get the remote file with a GET request executed in the page.
        Works in Chrome.
        Requires --allow-access-from-files and --disable-web-security options
        Got from https://stackoverflow.com/questions/47068912/how-to-download-a-file-using-the-remote-selenium-webdriver
        Raise DownloadFail is something got wrong
    """

    result = driver.execute_async_script("""
        var uri = arguments[0];
        var callback = arguments[1];
        var toBase64 = function(buffer) {
            for(var r, n =  new Uint8Array(buffer), t = n.length, a = new Uint8Array(4*Math.ceil(t/3)), i = new Uint8Array(64), o = 0, c = 0; 64 > c; ++c) {
                i[c] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/".charCodeAt(c);
            }
            for (c = 0; t - t % 3 > c; c += 3, o +=4) {
                r = n[c] << 16 | n[c+1] << 8 | n[c+2];
                a[o] = i[r >> 18];
                a[o + 1] = i[r >>12 & 63];
                a[o + 2] = i[r >> 6 & 63];
                a[o + 3] = i[63 & r];
            }

            if (t % 3 === 1) {
                r = n[t - 1];
                a[o] = i[r >> 2];
                a[o + 1] = i[r << 4 & 63];
                a[o + 2] = 61;
                a[o + 3] = 61
            }

            if (t % 3 === 2) {
                r=(n[t - 2] << 8) + n[t - 1];
                a[o] = i[r >> 10];
                a[o + 1] = i[r >> 4 & 63];
                a[o + 2] = i[r << 2 & 63];
                a[o + 3] = 61
            }
            return new TextDecoder("ascii").decode(a)
        };

        var xhr = new XMLHttpRequest();
        xhr.responseType = 'arraybuffer';
        xhr.onload = function(){ callback(toBase64(xhr.response)) };
        xhr.onerror = function(){ callback(xhr.status) };
        xhr.open('GET', uri);
        xhr.send();
    """, uri)

    if isinstance(result, int):
        raise DownloadFail('Request failed', result, driver.get_log('browser'))

    return base64.b64decode(result)


class AmadeusClient(object):
    LOGIN_PAGE = 'https://api.connect.travelaudience.com/customer/login/?next=/customer/home/'
    TIMEOUT = 10

    def __init__(self, driver, login, password, logger):
        self._driver = driver
        self._login = login
        self._password = password
        self._logger = logger

    def get_report(self):
        self.login()
        self.wait(self.TIMEOUT)
        self.select_all_fields_in_report()

        return self.download_report()

    def click_download_button(self):
        download_button = self._driver.find_element_by_xpath('//button[@class=\'btn btn-primary\']')
        download_button.click()

    def download_report(self):
        self.click_download_button()
        return self._read_downloaded_file()

    def login(self):
        from selenium.webdriver.common.keys import Keys
        from selenium.webdriver.support.ui import WebDriverWait

        self._driver.get(self.LOGIN_PAGE)
        # self.wait(self.TIMEOUT)

        login_finder = lambda _driver: _driver.find_element_by_id('id_username')
        login_elem = WebDriverWait(self._driver, 10, 1).until(login_finder)
        login_elem.send_keys(self._login)

        password_elem = self._driver.find_element_by_id('id_password')
        password_elem.send_keys(self._password)
        password_elem.send_keys(Keys.RETURN)

        self._logger.info('Login done')

    def select_all_fields_in_report(self):
        self._logger.info('Selecting all checkboxes in report')
        dropdown = self._driver.find_element_by_class_name('dropdown-toggle')
        dropdown.click()

        checkboxes = self._driver.find_elements_by_xpath("//input[@type='checkbox']")
        for checkbox in checkboxes:
            if not checkbox.is_selected():
                checkbox.click()

        self._logger.info('All checkboxes has been selected')

    def wait(self, time_to_wait):
        self._logger.info('Waiting for %d', time_to_wait)
        time.sleep(time_to_wait)

    def _read_downloaded_file(self):
        from selenium.webdriver.support.ui import WebDriverWait
        file_name = WebDriverWait(self._driver, 20, 1).until(get_file_names_chrome)
        self._logger.info('Ready to download file %s', file_name)

        content = get_file_content_chrome(self._driver, file_name)
        self._logger.info('Read %d bytes', len(content))

        return content


class AviaGetAmadeusReport(AviaBaseTask):
    """ Get Amadeus report """

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass  # We do not need caches

        environments = (
            PipEnvironment('yandex-yt', version='0.10.8'),
            PipEnvironment('yandex-yt-yson-bindings-skynet', version='0.3.32-0'),
            PipEnvironment('selenium'),
            PipEnvironment('raven'),
        )

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Secret Settings') as yt_settings:
            vaults_owner = sdk2.parameters.String('Token vault owner', required=True)
            yt_vault_name = sdk2.parameters.String('YT Token vault name', required=True, default='YT_TOKEN')
            amadeus_login = sdk2.parameters.String('Amadeus login', required=True)
            amadeus_vault_name = sdk2.parameters.String('Amadeus password vault name', required=True)

        with sdk2.parameters.Group('Task settings') as task_settings:
            yt_root_dir = sdk2.parameters.String(
                'YT Root directory',
                required=True,
                default='//home/avia/logs/avia-amadeus-marker-log',
            )

    def _create_selenium_driver(self):
        from selenium import webdriver
        from selenium.webdriver.remote.remote_connection import RemoteConnection
        chrome_options = webdriver.ChromeOptions()
        chrome_options.add_argument('--disable-web-security')
        chrome_options.add_argument('--allow-file-access-from-files')

        driver = webdriver.Remote(
            command_executor=RemoteConnection(
                SELENIUM_GRID_URL, resolve_ip=False
            ),
            desired_capabilities=BROWSER_CAPABILITIES,
            options=chrome_options,
        )

        driver.set_script_timeout(300000)
        return driver

    def on_execute(self):
        logging.info('Start')

        self._configure_sentry()

        amadeus_client = AmadeusClient(
            self._create_selenium_driver(),
            self.Parameters.amadeus_login,
            sdk2.Vault.data(self.Parameters.vaults_owner, self.Parameters.amadeus_vault_name),
            logging,
        )
        report = amadeus_client.get_report()

        self.write_report_to_yt(report)
        logging.info('End')

    def write_report_to_yt(self, report):
        import yt.wrapper as yt
        yt_client = YtClientFactory.create(
            'hahn',
            sdk2.Vault.data(self.Parameters.vaults_owner, self.Parameters.yt_vault_name),
        )

        yesterday = (datetime.today() - timedelta(1)).strftime('%Y-%m-%d')
        with yt_client.Transaction():
            target_path = yt.ypath_join(self.Parameters.yt_root_dir, yesterday)
            if yt_client.exists(target_path):
                yt_client.remove(target_path)

            records_to_write = []
            all_fields = set()
            for record in csv.DictReader(cStringIO.StringIO(report)):
                if record['DATE'] != yesterday:
                    continue

                new_record = {}
                for key, value in record.iteritems():
                    try:
                        if value is not None:
                            if key in INT_COLUMNS:
                                new_record[key] = int(value)

                            elif key in DOUBLE_COLUMNS:
                                new_record[key] = float(value.replace(',', '.'))

                            else:
                                new_record[key] = value
                        else:
                            new_record[key] = value
                    except ValueError:
                        logging.info('Bad value %r for key %r', value, key)
                        raise

                    all_fields.add(key)

                records_to_write.append(new_record)

            yt_client.create(
                'table',
                target_path,
                recursive=True,
                attributes={
                    'optimize_for': 'scan',
                    'schmea': generate_yt_schema_by_fields(all_fields),
                })

            yt_client.write_table(
                target_path,
                records_to_write,
            )

    def _configure_sentry(self):
        configure_logging(get_sentry_dsn(self))
