# -*- coding: utf-8 -*-
import logging
from datetime import datetime, timedelta

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.projects.avia.base import AviaBaseTask
from sandbox.projects.avia.lib.logs import configure_logging, get_sentry_dsn
from sandbox.projects.avia.lib.partner_cache import PartnerCache
from sandbox.projects.avia.lib.geo_point_cache import GeoPointCache
from sandbox.projects.avia.lib.yt_helpers import YtClientFactory
from sandbox.sandboxsdk.environments import PipEnvironment


class AviaImportMarker(AviaBaseTask):
    """ Base task of avia import marker tasks """

    class Requirements(sdk2.Task.Requirements):
        # https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
        cores = 1  # exactly 1 core
        ram = 8192  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

        environments = (
            PipEnvironment('raven'),
            PipEnvironment('requests'),
            PipEnvironment('yandex-yt', version='0.10.8'),
            PipEnvironment('yandex-yt-yson-bindings-skynet', version='0.3.32-0'),
        )
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Import parameters') as import_block:
            days = sdk2.parameters.Integer('Days', default_value=1)
            reference_yt_proxy = sdk2.parameters.String('Reference YT proxy', required=True, default_value='hahn')
            vaults_owner = sdk2.parameters.String('Vaults owner', required=True, default_value='AVIA')
            partner_code = sdk2.parameters.String('Partner\'s code', required=True)
            vaults_partner_code = sdk2.parameters.String('Vaults partner\'s code', default_value='')
            source = sdk2.parameters.String('Source', required=True)
            reference_yt_path = sdk2.parameters.String(
                'Reference YT path', required=True, default_value='//home/rasp/reference'
            )
            yt_partner_booking_root = sdk2.parameters.String(
                'YT partner booking root', required=True, default_value='//home/avia/logs/avia-partner-booking-log'
            )

    def on_prepare(self):
        configure_logging(
            sentry_dsn=get_sentry_dsn(self)
        )
        self._logger = logging.getLogger(__name__)
        self._right_date = datetime.now().date()
        self._left_date = self._right_date - timedelta(days=int(self.Parameters.days))
        self._yt = YtClientFactory.create(
            self.Parameters.reference_yt_proxy,
            sdk2.Vault.data(self.Parameters.vaults_owner, 'YT_TOKEN')
        )
        self.geo_point_cache = GeoPointCache(self._yt, self.Parameters.reference_yt_path)
        partner_cache = PartnerCache(self._yt, self.Parameters.reference_yt_path)
        self._partner = partner_cache.partner_by_code(self.Parameters.partner_code)

        super(AviaImportMarker, self).on_prepare()

    def _get_partner_secret(self, key):
        env_secrets = 'AVIA'
        partner_code = self.Parameters.vaults_partner_code or self.Parameters.partner_code
        secret_key = '_'.join([env_secrets, partner_code.upper(), key.upper()])
        return sdk2.Vault.data(self.Parameters.vaults_owner, secret_key)
