# -*- coding: utf-8 -*-
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.projects.avia.lib.marker import MarkerHTTPReader, MarkerWriter, MarkerTransfer
from sandbox.projects.avia.import_marker import AviaImportMarker


class AnyWayAnyDayMarkerReader(MarkerHTTPReader):
    def __init__(self, api_url, login, password, geo_point_cache, logger):
        """
        :type api_url: str
        :type login: str
        :type password: str
        """
        super(AnyWayAnyDayMarkerReader, self).__init__(
            logger=logger,
            statuses_map={
                'booking': 'booking',
                'paid': 'paid',
                'cancel': 'cancel',
            },
            geo_point_cache=geo_point_cache,
        )
        self._url = api_url
        self._login = login
        self._password = password

    def _get_request_data(self, report_date):
        return (
            self._url,
            {
                'partner': self._login,
                'password': self._password,
                'date_from': report_date.strftime('%Y-%m-%d'),
                'date_to': report_date.strftime('%Y-%m-%d'),
            }
        )

    def _get_request_auth(self):
        return None

    def parse_report(self, content):
        """
        AWAD order format example:
        {
            "status": "cancel",
            "orderid": "459763976",
            "price": 6569.23,
            "currency": "RUB",
            "created_at": "2020-04-06",
            "type": "RT",
            "from": "MOW",
            "to": "LED",
            "pax": 2
        }
        """
        import json

        result_orders = []
        try:
            orders = json.loads(content)
        except Exception:
            self._logger.exception('Bad anywayanyday Json')
            return []

        for order in orders:
            try:
                order = {
                    'order_id': order['orderid'],
                    'status': self._parse_status(order['status']),
                    'created_at': datetime.strptime(order['created_at'], '%Y-%m-%d'),
                    'price': order['price'],
                    'marker': order['marker'],
                    'flights': [],
                }
                result_orders.append(order)
            except Exception:
                self._logger.exception('Parse error')
        self._fillin_airports(orders)
        return result_orders


class AviaImportAnyWayAnyDayMarker(AviaImportMarker):
    """ Import marker from anywayanyday """

    class Parameters(AviaImportMarker.Parameters):
        with sdk2.parameters.Group('Import parameters') as import_block:
            api_url = sdk2.parameters.String('Report Url', required=True, default='https://api.anywayanyday.com/YandexReport')
            login = sdk2.parameters.String('Login', required=True, default='yandexprod2020')
            api_password_vault_key = sdk2.parameters.String(
                'Report API password vault key',
                required=True,
                default_value='AVIA_ANYWAYANYDAY_PASSWORD',
            )

    class Requirements(AviaImportMarker.Requirements):
        environments = AviaImportMarker.Requirements.environments.default

    def on_execute(self):
        api_url = self.Parameters.api_url
        login = self.Parameters.login
        password = sdk2.Vault.data(self.Parameters.vaults_owner, self.Parameters.api_password_vault_key)

        marker_transfer = MarkerTransfer(
            partner=self._partner,
            marker_writer=MarkerWriter(
                self.Parameters.source,
                self._logger,
                self.Parameters.yt_partner_booking_root,
                self._yt,
            ),
            marker_reader=AnyWayAnyDayMarkerReader(api_url, login, password, self.geo_point_cache, self._logger),
            logger=self._logger,
        )

        self._logger.info('Transferring date range: %s - %s', self._left_date, self._right_date)
        report_date = self._left_date

        while report_date <= self._right_date:
            marker_transfer.transfer(report_date)
            report_date += timedelta(days=1)

        self._logger.info('Stop: transferring data in date range')
