# -*- coding: utf-8 -*-
from datetime import datetime, timedelta

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.avia.lib.marker import MarkerHTTPReader, MarkerWriter, MarkerTransfer
from sandbox.projects.avia.import_marker import AviaImportMarker


class AviakassaMarkerReader(MarkerHTTPReader):
    REQUEST_URL = 'https://meta.aviakassa.com/v4/meta/avia/a/stat'

    def __init__(self, password, geo_point_cache, logger):
        """
        :type password: str
        """
        super(AviakassaMarkerReader, self).__init__(
            logger=logger,
            statuses_map={
                'PROCESSING': 'booking',
                'PAID': 'paid',
                'CANCELLED': 'cancel',
                'PARTIALLYCANCELLED': 'cancel',
            },
            geo_point_cache=geo_point_cache,
        )
        self._password = password

    def _get_request_data(self, report_date):
        return (
            self.REQUEST_URL,
            {
                'date1': report_date.strftime('%Y-%m-%d'),
                'date2': report_date.strftime('%Y-%m-%d'),
                'password': self._password,
            }
        )

    def _get_request_auth(self):
        return None

    def parse_report(self, content):
        from lxml import etree
        from sandbox.projects.avia.lib.safe_lxml import fromstring as safe_fromstring
        orders = []
        try:
            tree = safe_fromstring(content)
        except etree.XMLSyntaxError:
            self._logger.exception('Bad Aviakassa XML')
            return []

        for order_tree in tree.xpath('//bookings/booking'):
            try:
                order_dict = {row.tag: row.text for row in order_tree}
                if not order_dict:
                    continue
                flight_dicts = [
                    {
                        'from': flight['departure'],
                        'to': flight['arrival'],
                        'departure_dt': self._convert_dt(
                            flight['departureDate'] + ' ' + flight['departureTime']
                        ),
                        'arrival_dt': self._convert_dt(
                            flight['arrivalDate'] + ' ' + flight['arrivalTime']
                        ),
                    }
                    for flight in
                    [
                        {
                            param.tag: param.text for param in flight_tree
                        }
                        for flight_tree in order_tree.xpath('segment/flight')
                    ]
                ]
                order = {
                    'order_id': order_dict.get('id'),
                    'created_at': self._convert_dt(order_dict.get('created_at')),
                    'price': order_dict.get('price'),
                    'currency': order_dict.get('currency'),
                    'status': self._parse_status(order_dict.get('state')),
                    'marker': order_dict.get('marker'),
                    'flights': flight_dicts,
                }
                orders.append(order)
            except Exception:
                self._logger.exception('Parse error')
        self._fillin_airports(orders)
        return orders

    @staticmethod
    def _convert_dt(dt_str):
        return datetime.strptime(dt_str.split('.')[0], '%Y-%m-%d %H:%M:%S')


class AviaImportAviakassaMarker(AviaImportMarker):
    """ Import marker from Aviakassa """

    class Requirements(AviaImportMarker.Requirements):
        environments = AviaImportMarker.Requirements.environments.default + (
            PipEnvironment('lxml'),
        )

    def on_execute(self):
        password = self._get_partner_secret('password')

        marker_transfer = MarkerTransfer(
            partner=self._partner,
            marker_writer=MarkerWriter(
                self.Parameters.source,
                self._logger,
                self.Parameters.yt_partner_booking_root,
                self._yt,
            ),
            marker_reader=AviakassaMarkerReader(password, self.geo_point_cache, self._logger),
            logger=self._logger,
        )

        self._logger.info('Transferring date range: %s - %s', self._left_date, self._right_date)
        report_date = self._left_date
        while report_date <= self._right_date:
            marker_transfer.transfer(report_date)
            report_date += timedelta(days=1)

        self._logger.info('Stop: transferring data in date range')
