# -*- coding: utf-8 -*-
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.avia.import_marker import AviaImportMarker
from sandbox.projects.avia.lib.marker import MarkerHTTPReader, MarkerWriter, MarkerTransfer
from sandbox.projects.avia.lib.safe_lxml import fromstring as safe_fromstring


class AviaoperatorMarkerReader(MarkerHTTPReader):

    def __init__(self, password, url_to_fetch, geo_point_cache, logger):
        """
        :type password: str
        """
        super(AviaoperatorMarkerReader, self).__init__(
            logger=logger,
            statuses_map={
                'PROCESSING': 'booking',
                'PAID': 'paid',
                'CANCELLED': 'cancel',
            },
            geo_point_cache=geo_point_cache,
        )
        self._password = password
        self._url_to_fetch = url_to_fetch

    def _get_request_data(self, report_date):
        return (
            self._url_to_fetch,
            {
                'date1': report_date.strftime('%Y-%m-%d'),
                'date2': report_date.strftime('%Y-%m-%d'),
                'password': self._password,
            }
        )

    def _get_request_auth(self):
        return None

    def parse_report(self, content):
        from lxml import etree
        orders = []
        try:
            tree = safe_fromstring(content)
        except etree.XMLSyntaxError:
            self._logger.exception('Bad Aviaoperator XML')
            return []

        for order_tree in tree.xpath('//bookings/booking'):
            try:
                order_dict = {row.tag: row.text for row in order_tree}
                flight_dicts = [
                    {
                        'from': flight['departure'],
                        'to': flight['arrival'],
                        'departure_dt': self._convert_no_seconds_dt(
                            flight['departureDate'] + ' ' + flight['departureTime']
                        ),
                        'arrival_dt': self._convert_no_seconds_dt(
                            flight['arrivalDate'] + ' ' + flight['arrivalTime']
                        ),
                    }
                    for flight in
                    [
                        {
                            param.tag: param.text for param in flight_tree
                        }
                        for flight_tree in order_tree.xpath('segment/flight')
                    ]
                ]
                order = {
                    'order_id': order_dict.get('id'),
                    'created_at': self._convert_dt(order_dict.get('created_at')),
                    'price': order_dict.get('price'),
                    'currency': order_dict.get('currency'),
                    'status': self._parse_status(order_dict.get('state')),
                    'marker': order_dict.get('marker'),
                    'flights': flight_dicts,
                }
                orders.append(order)
            except Exception:
                self._logger.exception('Parse error')
        self._fillin_airports(orders)
        return orders

    @staticmethod
    def _convert_dt(dt_str):
        return datetime.strptime(dt_str.split('.')[0], '%Y-%m-%d %H:%M:%S')

    @staticmethod
    def _convert_no_seconds_dt(dt_str):
        return datetime.strptime(dt_str.split('.')[0], '%Y-%m-%d %H:%M')


class AviaImportAviaoperatorMarker(AviaImportMarker):
    """ Import marker from Aviaoperator """

    class Requirements(AviaImportMarker.Requirements):
        environments = AviaImportMarker.Requirements.environments.default + (
            PipEnvironment('lxml'),
        )

    class Parameters(AviaImportMarker.Parameters):
        with sdk2.parameters.Group('Import parameters') as import_block:
            url_to_fetch = sdk2.parameters.String(
                'URL to fetch stats from',
                required=False,
                default_value='http://api.aviaoperator.com/api/getstatistics')

    def on_execute(self):
        password = self._get_partner_secret('password')

        marker_transfer = MarkerTransfer(
            partner=self._partner,
            marker_writer=MarkerWriter(
                self.Parameters.source,
                self._logger,
                self.Parameters.yt_partner_booking_root,
                self._yt,
            ),
            marker_reader=AviaoperatorMarkerReader(
                password, self.Parameters.url_to_fetch, self.geo_point_cache, self._logger),
            logger=self._logger,
        )

        self._logger.info('Transferring date range: %s - %s', self._left_date, self._right_date)
        report_date = self._left_date
        while report_date <= self._right_date:
            marker_transfer.transfer(report_date)
            report_date += timedelta(days=1)

        self._logger.info('Stop: transferring data in date range')
