# -*- coding: utf-8 -*-
import hashlib
from collections import OrderedDict
from datetime import datetime, timedelta

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.avia.lib.marker import BaseMarkerReader, MarkerWriter, MarkerTransfer
from sandbox.projects.avia.import_marker import AviaImportMarker


class AzimuthMarkerReader(BaseMarkerReader):
    REQUEST_URL = 'https://booking.azimuth.aero/bitrix/components/travelshop/ibe.soap/travelshop_booking.php'
    LOGIN = 'Yandex_A4'

    def __init__(self, session_token, shared_secret, geo_point_cache, logger):
        super(AzimuthMarkerReader, self).__init__(
            logger=logger,
            statuses_map={},  # Useless here
            geo_point_cache=geo_point_cache,
        )
        self._session_token = session_token
        self._shared_secret = shared_secret

    def import_data(self, date):
        import requests
        self._logger.info('Process: %s', date.strftime('%Y-%m-%d'))
        (url, query_xml) = self._get_request_data(date)
        self._logger.info('Get %s', url)
        r = requests.post(
            url,
            headers={
                'SOAPAction': 'urn:#GetOrderList',
                'Content-Type': 'text/xml; charset=utf-8',
            },
            data=query_xml.encode('utf-8'),
        )
        orders = self.parse_report(r.content)
        for order in orders:
            if order['created_at'].date() == date:
                yield order

    def parse_report(self, content):
        from lxml import etree
        from sandbox.projects.avia.lib.safe_lxml import fromstring as safe_fromstring

        orders = []
        try:
            tree = safe_fromstring(content)
        except etree.XMLSyntaxError:
            self._logger.exception('Bad Azimuth XML')
            return orders

        try:
            for order_element in tree.xpath('//ns1:orders/ns1:GetOrderListOrder', namespaces=tree.nsmap):
                flights = [
                    {
                        'from': flight.findtext('ns1:departure_airport_code', namespaces=tree.nsmap),
                        'to': flight.findtext('ns1:arrival_airport_code', namespaces=tree.nsmap),
                        'departure_dt': self._convert_dt(
                            flight.findtext('ns1:departure_date', namespaces=tree.nsmap),
                            flight.findtext('ns1:departure_time', namespaces=tree.nsmap),
                        ),
                        'arrival_dt': self._convert_dt(
                            flight.findtext('ns1:arrival_date', namespaces=tree.nsmap),
                            flight.findtext('ns1:arrival_time', namespaces=tree.nsmap),
                        ),
                    }
                    for flight in
                    order_element.xpath('ns1:segments/ns1:AirSegment', namespaces=tree.nsmap)
                ]

                order = {
                    'order_id': order_element.findtext('ns1:order_id', namespaces=tree.nsmap),
                    'price': order_element.findtext('ns1:full_price', namespaces=tree.nsmap),
                    'marker': order_element.findtext('ns1:partner_data', namespaces=tree.nsmap),
                    'flights': flights,

                    'created_at': self._convert_dt(
                        *order_element.findtext('ns1:booking_timestamp', namespaces=tree.nsmap).split()  # 13.07.2019 14:27:20
                    ),
                    'status': self._parse_status(order_element),
                }

                ticketing_timestamp = order_element.findtext(
                    'ns1:ticketing_timestamp', namespaces=tree.nsmap
                )
                if ticketing_timestamp:
                    order['confirm_date'] = self._convert_dt(*ticketing_timestamp.split())
                orders.append(order)
        except Exception:
            self._logger.exception('Parse error')

        self._fillin_airports(orders)
        return orders

    def _get_request_data(self, report_date):
        next_day = report_date + timedelta(days=1)
        dt_fmt = '%d.%m.%Y %H:%M:%S'
        params = OrderedDict((
            ('session_token', self._session_token),
            ('timestamp_from', report_date.strftime(dt_fmt)),
            ('timestamp_to', next_day.strftime(dt_fmt)),
        ))

        m = hashlib.md5()
        m.update(''.join(params.values()) + self._shared_secret)
        params['hash'] = m.hexdigest()

        query_xml = """
            <soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/" xmlns:tais="http://www.tais.ru/">
               <soapenv:Header/>
               <soapenv:Body>
                  <tais:GetOrderListInput>
                     <tais:session_token>{session_token}</tais:session_token>
                     <tais:hash>{hash}</tais:hash>
                     <tais:timestamp_from>{timestamp_from}</tais:timestamp_from>
                     <tais:timestamp_to>{timestamp_to}</tais:timestamp_to>
                  </tais:GetOrderListInput>
               </soapenv:Body>
            </soapenv:Envelope>
        """.format(**params)
        return self.REQUEST_URL, query_xml

    def _parse_status(self, order):
        if order.findtext('ns1:canceled', namespaces=order.nsmap) == 'Y':
            return 'cancel'
        elif order.findtext('ns1:finalized', namespaces=order.nsmap) == 'Y':
            return 'paid'
        else:
            return 'booking'

    @staticmethod
    def _convert_dt(date, time):
        """
            param str date: 14.07.2019
            param str time: 09:45:00
        """
        return datetime.strptime(date + time, '%d.%m.%Y%H:%M:%S')


class AviaImportAzimuthMarker(AviaImportMarker):
    """ Import marker from Azimuth """

    class Requirements(AviaImportMarker.Requirements):
        environments = AviaImportMarker.Requirements.environments.default + (
            PipEnvironment('lxml'),
        )

    def on_execute(self):
        session_token = self._get_partner_secret('session_token')
        shared_secret = self._get_partner_secret('shared_secret')
        marker_transfer = MarkerTransfer(
            partner=self._partner,
            marker_writer=MarkerWriter(
                self.Parameters.source,
                self._logger,
                self.Parameters.yt_partner_booking_root,
                self._yt,
            ),
            marker_reader=AzimuthMarkerReader(session_token, shared_secret, self.geo_point_cache, self._logger),
            logger=self._logger,
        )

        self._logger.info('Transferring date range: %s - %s', self._left_date, self._right_date)
        report_date = self._left_date
        while report_date <= self._right_date:
            marker_transfer.transfer(report_date)
            report_date += timedelta(days=1)

        self._logger.info('Stop: transferring data in date range')
