# -*- coding: utf-8 -*-
from datetime import datetime, timedelta

from sandbox.projects.avia.import_marker import AviaImportMarker
from sandbox.projects.avia.lib.logs import remove_secrets
from sandbox.projects.avia.lib.marker import MarkerHTTPReader, MarkerTransfer, MarkerWriter
from sandbox.sandboxsdk.environments import PipEnvironment


class BiletdvMarkerReader(MarkerHTTPReader):
    REQUEST_URL = 'https://avia.biletdv.ru/DesktopModules/SASPartnerSales/GetPartnerSalesData.aspx'
    LOGIN = 'yandex'

    def __init__(self, password, geo_point_cache, logger):
        """
        :param str password: пароль для доступа
        """
        super(BiletdvMarkerReader, self).__init__(
            logger=logger,
            statuses_map={
                'SALE': 'paid',
                'REFUND': 'cancel',
            },
            geo_point_cache=geo_point_cache,
        )
        self._password = password

    def _get_request_data(self, report_date):
        yesterday = report_date - timedelta(days=1)
        tomorrow = report_date + timedelta(days=1)
        return (
            self.REQUEST_URL,
            {
                'date1': yesterday.strftime('%d.%m.%Y'),
                'date2': tomorrow.strftime('%d.%m.%Y'),
                'userName': self.LOGIN,
                'password': self._password,
            }
        )

    def _get_request_auth(self):
        return None

    def import_data(self, date):
        import requests

        self._logger.info('Process: %s', date.strftime('%Y-%m-%d'))
        (url, params) = self._get_request_data(date)
        self._logger.info('Get %s, %s', url, remove_secrets(params))
        r = requests.get(
            url,
            auth=self._get_request_auth(),
            params=params,
            verify=False,  # RASPTICKETS-20297
        )
        orders = self.parse_report(r.content)
        for order in orders:
            if order['created_at'].date() == date:
                yield order

    def parse_report(self, content):
        from lxml import etree
        from sandbox.projects.avia.lib.safe_lxml import fromstring as safe_fromstring

        try:
            tree = safe_fromstring(content)
        except etree.XMLSyntaxError:
            self._logger.exception('Bad BiletDV XML')
            return

        for order_tree in tree.xpath('//Record'):
            try:
                order_dict = {row.tag: row.text for row in order_tree}
                if order_tree.get('Type') == 'REFUND':
                    continue
                order = {
                    'order_id': order_dict.get('PNR'),
                    'created_at': self._convert_dt(order_dict.get('CreatedUTC')),
                    'price': order_dict.get('TotalSumma'),
                    'status': self._parse_status(order_tree.get('Type')),
                    'marker': order_dict.get('PersonID'),
                    'airport_from': order_dict.get('DepCity'),
                    'airport_to': order_dict.get('DestCity'),
                    'ticket_number': order_dict.get('TicketNumber'),
                }
                yield order
            except Exception:
                self._logger.exception('Parse error')

    @staticmethod
    def _convert_dt(dt_str):
        return datetime.strptime(dt_str, '%Y-%m-%dT%H:%M:%S')


class AviaImportBiletdvMarker(AviaImportMarker):
    """ Import marker from Biletdv """

    class Requirements(AviaImportMarker.Requirements):
        environments = AviaImportMarker.Requirements.environments.default + (
            PipEnvironment('lxml'),
        )

    def on_execute(self):
        password = self._get_partner_secret('password')

        marker_transfer = MarkerTransfer(
            partner=self._partner,
            marker_writer=MarkerWriter(
                self.Parameters.source,
                self._logger,
                self.Parameters.yt_partner_booking_root,
                self._yt,
            ),
            marker_reader=BiletdvMarkerReader(password, self.geo_point_cache, self._logger),
            logger=self._logger,
        )

        self._logger.info('Transferring date range: %s - %s', self._left_date, self._right_date)
        report_date = self._left_date
        while report_date <= self._right_date:
            marker_transfer.transfer(report_date)
            report_date += timedelta(days=1)

        self._logger.info('Stop: transferring data in date range')
