# -*- coding: utf-8 -*-
from datetime import datetime, timedelta

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.avia.lib.marker import MarkerHTTPReader, MarkerWriter, MarkerTransfer
from sandbox.projects.avia.import_marker import AviaImportMarker


class BiletixMarkerReader(MarkerHTTPReader):
    REQUEST_URL = 'http://partners.biletix.ru/api/export_orders_yt/ordersYandex.json'

    def __init__(self, geo_point_cache, logger):
        super(BiletixMarkerReader, self).__init__(
            logger=logger,
            statuses_map={
                'booked': 'booking',
                'paid': 'paid',
                'cancelled': 'cancel',
            },
            geo_point_cache=geo_point_cache
        )
        # biletix responds with data for last 60 days
        self.received_data = None

    def import_data(self, date):
        """
        Собираем данные от партнера и сохраняем в промежуточную структуру
        Формат стурктуры смотри в parse_report

        :type date: datetime.date
        :param date: дата, на которую нужно забрать данные
        :rtype: list[dict[str, Any]]
        :return: формат возвращаемой структуры смотри в parse_report
        """
        if not self.received_data:
            self.received_data = super(BiletixMarkerReader, self).import_data(date)
        return self.filter_by_date(date)

    def filter_by_date(self, date):
        """
        Фильтруем данные для определенной даты.
        Фильтрация производится по полю created_at во внутреннем словаре

        :param datetime.date date: дата фильтрации
        :return: отфильтрованные по дате данные
        """
        assert self.received_data
        return filter(lambda item: item.get('created_at').date() == date, self.received_data)

    def _get_request_data(self, report_date):
        return self.REQUEST_URL, {}

    def _get_request_auth(self):
        return None

    def parse_report(self, content):
        import ujson

        orders = []
        try:
            order_list = ujson.loads(content)
        except ValueError:
            self._logger.exception('Bad Biletix JSON')
            return []

        for order_dict in order_list:
            try:
                price, currency = self._parse_price(order_dict.get('price'))
                order = {
                    'order_id': order_dict.get('order_id'),
                    'created_at': self._convert_dt(order_dict.get('date_create')),
                    'price': price,
                    'currency': currency,
                    'status': self._parse_status(order_dict.get('status')),
                    'marker': order_dict.get('marker'),
                    'airport_from': order_dict.get('departure'),
                    'airport_to': order_dict.get('arrival'),
                    'trip_type': self._parse_owrt(order_dict.get('owrt')),
                }
                orders.append(order)
            except Exception:
                self._logger.exception('Parse error')
        return orders

    def _convert_dt(self, dt_str):
        return datetime.strptime(dt_str, '%Y-%m-%d %H:%M:%S')

    def _parse_owrt(self, owrt):
        owrt_map = {
            'OW': 'oneway',
            'RT': 'roundtrip',
            'OJ': 'openjaw',
        }
        return owrt_map.get(owrt)

    def _parse_price(self, input_str):
        """
        Разбиваем строчку с ценой на цену и валюту
        :param basestring input_str: строка вида 1123.01 RUR
        :return: кортеж вида (float_price, currency)
        """
        price_str, currency_str = input_str.split(' ')
        price = float(price_str)
        return price, currency_str


class AviaImportBiletixMarker(AviaImportMarker):
    """ Import marker from Biletix """

    class Requirements(AviaImportMarker.Requirements):
        environments = AviaImportMarker.Requirements.environments.default + (
            PipEnvironment('ujson'),
        )

    def on_execute(self):
        marker_transfer = MarkerTransfer(
            partner=self._partner,
            marker_writer=MarkerWriter(
                self.Parameters.source,
                self._logger,
                self.Parameters.yt_partner_booking_root,
                self._yt,
            ),
            marker_reader=BiletixMarkerReader(self.geo_point_cache, self._logger),
            logger=self._logger,
        )

        self._logger.info('Transferring date range: %s - %s', self._left_date, self._right_date)
        report_date = self._left_date
        while report_date <= self._right_date:
            marker_transfer.transfer(report_date)
            report_date += timedelta(days=1)

        self._logger.info('Stop: transferring data in date range')
