# -*- coding: utf-8 -*-
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.avia.lib.marker import MarkerHTTPReader, MarkerWriter, MarkerTransfer
from sandbox.projects.avia.import_marker import AviaImportMarker


class CitytravelMarkerReader(MarkerHTTPReader):
    REQUEST_URL = 'http://avia.city.travel/aviasales/statistics'

    def __init__(self, login, password, geo_point_cache, logger):
        """
        :param str password: пароль для доступа
        """
        super(CitytravelMarkerReader, self).__init__(
            logger=logger,
            statuses_map={
                'PROCESSING': 'booking',
                'PAID': 'paid',
                'CANCELLED': 'cancel',
            },
            geo_point_cache=geo_point_cache,
        )
        self._password = password
        self._login = login

    def _get_request_data(self, report_date):
        return (
            self.REQUEST_URL,
            {
                'date1': report_date.strftime('%Y-%m-%d'),
                'date2': report_date.strftime('%Y-%m-%d'),
                'partner': self._login,
                'password': self._password,
            }
        )

    def _get_request_auth(self):
        import requests.auth
        return requests.auth.HTTPBasicAuth(self._login, self._password)

    def parse_report(self, content):
        from lxml import etree
        from sandbox.projects.avia.lib.safe_lxml import fromstring as safe_fromstring

        orders = []
        try:
            tree = safe_fromstring(content)
        except etree.XMLSyntaxError:
            self._logger.exception('Bad CITYTRAVEL XML')
            return []

        for order_tree in tree.xpath('//bookings/booking'):
            try:
                order_dict = {row.tag: row.text for row in order_tree}
                order = {
                    'order_id': order_dict.get('id'),
                    'created_at': self._convert_dt(order_dict.get('created_at')),
                    'price': order_dict.get('price'),
                    'status': self._parse_status(order_dict.get('state')),
                    'marker': order_dict.get('marker'),
                    'currency': order_dict.get('currency'),
                }
                orders.append(order)
            except Exception:
                self._logger.exception('Parse error')
        return orders

    def _convert_dt(self, dt_str):
        return datetime.strptime(dt_str.split('.')[0], '%Y-%m-%d %H:%M:%S')


class AviaImportCitytravelMarker(AviaImportMarker):
    """ Import marker from Citytravel """

    class Requirements(AviaImportMarker.Requirements):
        environments = AviaImportMarker.Requirements.environments.default + (
            PipEnvironment('lxml'),
        )

    class Parameters(AviaImportMarker.Parameters):
        login = sdk2.parameters.String('Login', required=True, default='yandex')

    def on_execute(self):
        password = self._get_partner_secret('password')

        marker_transfer = MarkerTransfer(
            partner=self._partner,
            marker_writer=MarkerWriter(
                self.Parameters.source,
                self._logger,
                self.Parameters.yt_partner_booking_root,
                self._yt,
            ),
            marker_reader=CitytravelMarkerReader(self.Parameters.login, password, self.geo_point_cache, self._logger),
            logger=self._logger,
        )

        self._logger.info('Transferring date range: %s - %s', self._left_date, self._right_date)
        report_date = self._left_date
        while report_date <= self._right_date:
            marker_transfer.transfer(report_date)
            report_date += timedelta(days=1)

        self._logger.info('Stop: transferring data in date range')
