# -*- coding: utf-8 -*-
import json
import urllib
from datetime import timedelta

from sandbox import sdk2
from sandbox.projects.avia.import_marker import AviaImportMarker
from sandbox.projects.avia.lib.marker import MarkerHTTPReader, MarkerTransfer, MarkerWriter
from sandbox.sandboxsdk.environments import PipEnvironment


class ClickaviaMarkerReader(MarkerHTTPReader):
    REQUEST_URL_TEMPLATE = 'https://www.clickavia.ru/api/partners/{token}/orders'

    def __init__(self, geo_point_cache, logger, token):
        super(ClickaviaMarkerReader, self).__init__(
            statuses_map={
                'processing': 'booking',
                'canceled': 'cancel',
                'confirmed': 'paid',
                'paid': 'paid',
            },
            logger=logger,
            geo_point_cache=geo_point_cache,
        )
        self.REQUEST_URL = self.REQUEST_URL_TEMPLATE.format(
            token=urllib.quote(token)
        )

    def import_data(self, date):
        orders = super(ClickaviaMarkerReader, self).import_data(date)
        for order in orders:
            if order['created_at'].date() == date:
                yield order

    def parse_report(self, content):
        try:
            orders = json.loads(content)
        except ValueError:
            self._logger.error('Can not load JSON from content %s ...', content[:80])
            raise

        if not isinstance(orders, list):
            return

        for order in orders:
            (airport_from, airport_to), trip_type = self.get_airports(order['direction'].split('/'))

            yield {
                'order_id': str(order['id']),
                'created_at': self._convert_dt(order['created_at']),
                'status': self._parse_status(order['state']),
                'marker': order['subagent'],
                'price': float(order['sum']),
                'currency': order['currency'],
                'airport_from': airport_from,
                'airport_to': airport_to,
                'trip_type': trip_type,
            }

    @staticmethod
    def get_airports(directions):
        airport_from = directions[0]
        airport_to = directions[len(directions) / 2]

        trip_type = 'oneway'
        if len(directions) > 2:
            if directions[0] == directions[-1]:
                trip_type = 'roundtrip'
            else:
                trip_type = 'openjaw'

        return (airport_from, airport_to), trip_type

    @staticmethod
    def _convert_dt(dt_str):
        from parse import parse
        return parse('{:ti}', dt_str)[0]

    def _get_request_auth(self):
        return None

    def _get_request_data(self, report_date):
        date_from = report_date - timedelta(days=1)
        date_to = report_date + timedelta(days=1)
        return (
            self.REQUEST_URL,
            {
                'date_from': date_from.strftime('%Y-%m-%d'),
                'date_to': date_to.strftime('%Y-%m-%d'),
            }
        )


class AviaImportClickaviaMarker(AviaImportMarker):
    """ Import marker from Clickavia """

    class Requirements(AviaImportMarker.Requirements):
        environments = AviaImportMarker.Requirements.environments.default + (
            PipEnvironment('parse'),
        )

    class Parameters(AviaImportMarker.Parameters):
        with sdk2.parameters.Group('Import parameters') as import_block:
            partner_code = sdk2.parameters.String('Partner\'s code', required=True, default='clickaviatravelab')
            source = sdk2.parameters.String('Source', required=True, default='clickavia')

    def on_execute(self):
        marker_transfer = MarkerTransfer(
            partner=self._partner,
            marker_writer=MarkerWriter(
                self.Parameters.source,
                self._logger,
                self.Parameters.yt_partner_booking_root,
                self._yt,
            ),
            marker_reader=ClickaviaMarkerReader(
                geo_point_cache=self.geo_point_cache,
                logger=self._logger,
                token=sdk2.Vault.data(self.Parameters.vaults_owner, 'AVIA_CLICKAVIA_TOKEN'),
            ),
            logger=self._logger,
        )

        self._logger.info('Transferring date range: %s - %s', self._left_date, self._right_date)
        report_date = self._left_date
        while report_date <= self._right_date:
            marker_transfer.transfer(report_date)
            report_date += timedelta(days=1)

        self._logger.info('Stop: transferring data in date range')
